C
c**************************************************************************
c
c    MODOFC - a program to solve Optimal Flow Control problems
c                using USGS MODFLOW-96 to simulate groundwater flow
c
c                      Version 3.0         
c                      August 2006
c
c       Copyright 1998-2006 by David P. Ahlfeld
c
c       Contact:  David P. Ahlfeld
c                 Dept. of Civil and Environmental Engineering
c                 University of Massachusetts, Amherst, MA, 01003, USA
c                 413-545-2681
c                 ahlfeld@ecs.umass.edu
c
c
c ----------------------------------------------------------------------
c  
c                  NOTICE TO USERS OF THIS COMPUTER CODE
c                 -----------------------------------------
c    DISCLAIMER OF WARRANTY
c
c    No responsibility is assumed by the code owner or by The 
c    University of Massachusetts for any errors, mistakes,
c    or misrepresentations that may occur from the use of
c    this computer code. The user of this code accepts and uses the
c    code as it is, at the user's own risk, without reliance upon any
c    representation or description concerning the computer code.
c    No warranties of any kind, whether statutory, written, oral,
c    expressed, or implied (including warranties of fitness and
c    merchantability) shall apply.
c  
c    TERMS OF COMPUTER CODE USE AND DISTRIBUTION
c
c 1) The copyright holder and code manager of the MODOFC computer code is
c        David P. Ahlfeld of the University of Massachusetts      
c 2) Any person who recieves or uses this computer code should 
c        register with the code manager by email to ahlfeld@ecs.umass.edu
c 3) Any payments made to obtain this code represent charges for
c        duplication, handling and registration.
c 4) This code or any portion thereof may not be sold or otherwise distributed
c        for profit without the express written permission
c        of the code manager.
c 5) The registered person may make minor modification to this code necessary 
c        for use in specific applications. Such modifications constitute
c        creation of a new version of MODOFC to which all restrictions
c        described herein apply. 
c 6) The user of this code agrees to appropriately acknowledge the authors
c        of this code in publications that result from use of this
c        software or in products that include the use of this software
c 7) By acceptance of delivery of this computer code the registered person
c        acknowledges and understands the terms of the use
c        and distribution of this code. 
c
c=========================================================================
C
      MODULE OFCMOD
C
C----FILE UNIT NUMBERS  
      INTEGER OPTINF,TEMPF,MODNME,INGWM,IOUTG
c      tempf   -unit number for a temporary file used by MODOFC.  It is opened
c               and closed in subroutines GETMOD and CALMOD
c      optinf  -unit number for file 'opt.in', the optimization input
c      modnme  -unit number for the MODFLOW names file.  The name of this file
c                 is defined by the user in opt.in
c      ingwm   -unit number for the GWM file.
c      ioutg   -unit number for the GWM 'global' output file
c
      PARAMETER (IOUTG=93)
      PARAMETER (INGWM=94)
      PARAMETER (TEMPF=95)
	PARAMETER (OPTINF=96)
	PARAMETER (MODNME=97)
C
C-----MODFLOW VARIABLES NEEDED BY SUBROUTINE GETMOD
C     MODFLOW-96 HARD DIMENSIONS PERLEN AT 1000, LAYCON AT 200 AND CUNIT AT 40
      DIMENSION PERLEN(1000)
      COMMON /FLWCOM/LAYCON(200)
      CHARACTER*4 CUNIT(40) 
C     DEFINE CONTENTS OF MODFLOW ARRAY CUNIT HERE
      DATA CUNIT/'BCF ','WEL ','DRN ','RIV ','EVT ','TLK ','GHB ',
     1           'RCH ','SIP ','DE4 ','SOR ','OC  ','PCG ','GFD ',
     2           '    ','HFB ','RES ','STR ','IBS ','CHD ','    ',
     3           '    ','    ','    ','    ','    ','    ','    ',
     4           '    ','    ','    ','    ','    ','    ','    ',
     5           '    ','    ','    ','    ','    '/
     	END MODULE OFCMOD 

c--------------------------------------------------------------------------
C GWM: List all GWM subroutines to be called in this program
      USE GWM1BAS1SUBS, ONLY: GWM1BAS1AR,GWM1BAS1RW,GWM1BAS1RPP 
      USE GWM1RMS1SUBS, ONLY: GWM1RMS1PL,GWM1RMS1PP,GWM1RMS1FP, 
     &                        GWM1RMS1FM,GWM1RMS1AP,GWM1RMS1OT,
     &                        GWM1RMS1OS   
      USE GWM1DCV1, ONLY: GWF1DCV1FM,GWF1DCV1BD 
	USE OFCMOD
C-----GWM VARIABLES
      LOGICAL FIRSTSIM,LASTSIM,FINISH,MFCNVRG,GWMCNVRG  
      INTEGER IPERT,NPERT              
      INTEGER SLPITCNT
C-----MODFLOW-96 VARIABLES
      INTEGER NPER,NROW,NCOL,NLAY,ILSTFIL
      DOUBLE PRECISION HCLOSE
      INTEGER LDXM1,LDPXM1
      parameter (ldpxm1=1000000) ! space used for MODFLOW X array
      DOUBLE PRECISION xm1(ldpxm1) 
C-----LOCAL VARIABLES
	REAL HCLOSES
      CHARACTER*200 FNAME
c++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
C
C-----OPEN PRIMARY FILES
C
C     THE OPTIN FILE CONTAINS THE GWM OUTPUT FILE, MODFLOW NAMES FILE 
C         AND THE NAME OF THE GWM FILE
      OPEN (UNIT=OPTINF,FILE='OPT.IN',  STATUS='OLD')

C     OPEN THE GWM OUTPUT FILE
	READ(OPTINF,*)FNAME            ! READ NAME FROM OPT.IN
      OPEN (UNIT=IOUTG,FILE=FNAME)
      WRITE(IOUTG,1000) IOUTG,FNAME
C
C     THE NAME OF THE MODFLOW NAMES FILE IS READ FROM OPT.IN AND OPENED IN SUBROUTINE GETMOD 
      call getmod(NROW,NCOL,NLAY,NPER,ILSTFIL,HCLOSE)  ! get information from MODFLOW
      WRITE(IOUTG,1010) MODNME,FNAME
C
C     READ GWM NAMES FILE AND OPEN IT
	READ(OPTINF,*)FNAME
      OPEN (UNIT=INGWM,FILE=FNAME,STATUS='OLD')
      WRITE(IOUTG,1020) INGWM,FNAME
C
C     WRITE HEADER
      WRITE(IOUTG,1050) 
      ldxm1  = ldpxm1
      LASTSIM = .FALSE.
	FINISH = .FALSE.
	FIRSTSIM = .TRUE.
C-----------------------------------------------------------------C
C-----READ GWM INPUT
      hcloses = hclose   ! convert to single precision
      CALL GWM1BAS1AR(INGWM,ILSTFIL,IOUTG,NROW,NCOL,NLAY,NPER,
     &                PERLEN,HCLOSEs)
C
C-----BEGIN ITERATION LOOP FOR SOLVING MANAGEMENT PROBLEM
      DO 300 WHILE (.NOT. FINISH)
C
C-------PREPARE LOOP FOR SIMULATION RESPONSE
        CALL GWM1RMS1PL(IPERT,NPERT,FIRSTSIM,LASTSIM)
C
        DO 200 WHILE (IPERT .LE. NPERT)
C
C-------PREPARE CALCULATION OF SIMULATION RESPONSE
          CALL GWM1RMS1PP(ILSTFIL,MFCNVRG,IPERT,FIRSTSIM,LASTSIM)
C
C-------CALL MODFLOW 96
          CALL CALMOD(XM1,LDXM1,ILSTFIL,1,
     &                SLPITCNT,MFCNVRG,IPERT,LASTSIM)
          FIRSTSIM = .FALSE.
C
C-------ANALYZE AND RECORD SIMULATION RESPONSE
          CALL GWM1RMS1FP(MFCNVRG,IPERT,NPERT,FIRSTSIM,LASTSIM)
  200   ENDDO
C
C-------IF NOT THE LAST SIMULATION, SOLVE GWM PROBLEM AT THIS ITERATION
        IF(.NOT. LASTSIM)THEN           ! GWM PROCESS MUST BE ACTIVE
          CALL GWM1RMS1FM               ! FORMULATE THE GWM PROBLEM
          CALL GWM1RMS1AP(GWMCNVRG)     ! SOLVE THE GWM PROBLEM
          IF(.NOT.GWMCNVRG)CALL GWM1RMS1OT(-1)! WRITE GWM SOLUTION OUTPUT 
        ENDIF
C
C-------IF GWM PROBLEM HAS CONVERGED, ONE MORE SIMULATION RUN IS NEEDED
        IF(GWMCNVRG .AND. .NOT. LASTSIM)THEN
          CALL GWM1RMS1OT(0)            ! WRITE GWM SOLUTION OUTPUT 
          LASTSIM = .TRUE.              ! EXECUTE ONE LAST SIMULATION
C
C-------IF GWM HAS COMPLETED ITS LAST SIMULATION OR GWM IS NOT ACTIVE
        ELSEIF(LASTSIM)THEN
          FINISH = .TRUE.               ! TERMINATE ITERATION LOOP
        ENDIF
C
C-----END OF GWM ITERATION LOOP
  300 ENDDO
C
      STOP
 1000 FORMAT(1X,/1X,'OPEN MODOFC OUTPUT FILE',/,
     1  ' ON UNIT ',I4,':',/1X,A200)
 1010 FORMAT(1X,/1X,'OPEN MODFLOW NAME FILE',/,
     1  ' ON UNIT ',I4,':',/1X,A200)
 1020 FORMAT(1X,/1X,'OPEN GWM INPUT FILE',/,
     1  ' ON UNIT ',I4,':',/1X,A200)
 1050 FORMAT(1X,/1X,'MODOFC USES GWM INPUT AND OUTPUT FORMATS.',/,
     1 '   THE FOLLOWING OUTPUT IS GENERATED BY GWM SUBROUTINES ',/,
     2 '   AND MAKES REFERENCES TO THE RESULTS OF GWM',/,
     3 '   REFER TO THE GWM DOCUMENTATION FOR INTERPRETATION OF OUTPUT')
      END
c
c*************************************************************************
C IGETUNIT, URDCOM AND USTOP ARE BORROWED FROM MF2K
C
      SUBROUTINE USTOP(STOPMESS)
C     ******************************************************************
C     STOP PROGRAM, WITH OPTION TO PRINT MESSAGE BEFORE STOPPING
C     ******************************************************************
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      CHARACTER STOPMESS*(*)
C     ------------------------------------------------------------------
   10 FORMAT(1X,A)

      IF (STOPMESS.NE.' ') THEN
        WRITE(*,10) STOPMESS
      ENDIF
      STOP
      END

      SUBROUTINE URDCOM(IN,IOUT,LINE)
C
C-----VERSION 02FEB1999 URDCOM
C     ******************************************************************
C     READ COMMENTS FROM A FILE AND PRINT THEM.  RETURN THE FIRST LINE
C     THAT IS NOT A COMMENT
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      CHARACTER*(*) LINE
C     ------------------------------------------------------------------
   10 READ(IN,'(A)') LINE
      IF(LINE(1:1).NE.'#') RETURN
      L=LEN(LINE)
      IF(L.GT.79) L=79
      DO 20 I=L,1,-1
      IF(LINE(I:I).NE.' ') GO TO 30
   20 CONTINUE
   30 IF (IOUT.GT.0) WRITE(IOUT,'(1X,A)') LINE(1:I)
      GO TO 10
C
      END
C
      INTEGER FUNCTION IGETUNIT(IFIRST,MAXUNIT)
C     VERSION 19981030 ERB
C     ******************************************************************
C     FIND FIRST UNUSED FILE UNIT NUMBER BETWEEN IFIRST AND MAXUNIT
C     ******************************************************************
C        SPECIFICATIONS:
C     -----------------------------------------------------------------
      INTEGER I, IFIRST, IOST, MAXUNIT
      LOGICAL LOP
C     -----------------------------------------------------------------
C
      LOP = .TRUE.
C
C     LOOP THROUGH RANGE PROVIDED TO FIND FIRST UNUSED UNIT NUMBER
      DO 10 I=IFIRST,MAXUNIT
        INQUIRE(UNIT=I,IOSTAT=IOST,OPENED=LOP,ERR=5)
        IF (IOST.EQ.0) THEN
          IF (.NOT.LOP) THEN
            IGETUNIT = I
            RETURN
          ENDIF
        ENDIF
 5      CONTINUE
10    CONTINUE
C
C     IF THERE ARE NO UNUSED UNIT NUMBERS IN RANGE PROVIDED, RETURN
C     A VALUE INDICATING AN ERROR
      IGETUNIT = -1
C
      RETURN
      END
C
C GETMOD IS REQUIRED FOR MODOFC  
C
c************************************************************
      SUBROUTINE getmod(NROW,NCOL,NLAY,NPER,ILSTFIL,HCLOSE)
c************************************************************
c
c  purpose - read MODFLOW files to obtain information
c
c    output:
c      MODFLOW variables: nrow, ncol, nlay, nper, perlen
c      HCLOSE - MODFLOW convergence parameter
c      bot    - bottom elevation of each layer if problem unconfined
c----------------------------------------------------------------------------
	USE OFCMOD,   ONLY : tempf,optinf,modnme
	USE OFCMOD,   ONLY : laycon,PERLEN,CUNIT
      implicit double precision (a-h,o-z)
      DOUBLE PRECISION, ALLOCATABLE :: TMP(:)
	INTEGER nper,nrow,ncol,nlay,ILSTFIL,ITMUNI
      dimension djunk(2)
      integer iunit(40)
      character*24 tmp2
      character*60 names
      character*80 headng(2)
      DOUBLE PRECISION HCLOSE
c++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
C
c--- read in and open MODFLOW names file
      read(optinf,*) names
      open(unit=modnme,file=names,status='old',err=900)
c
c--- as MODFLOW files are read write contents to temp file
	open(unit=tempf,file='temp')
c
c--- open MODFLOW files and obtain MODFLOW variables: nrow,ncol,nlay,nper,iunit
	ilstfil=0
      call bas5df(isum,headng,nper,itmuni,totim,ncol,nrow,nlay,nodes,
     &    inbas,tempf,iunit,cunit,modnme,ixsec,ichflg,ifrefm,ilstfil)
C
C-----ALLOCATE ARRAYS BASED ON SIZE OF MODFLOW PROBLEM
	ALLOCATE (TMP(nrow*ncol*nlay))
c
c--- read past items 5,6,7,8 in BAS file to obtain perlen
      read(inbas,*)
      do 100 i=1,nlay
        call u2dint(tmp,tmp2,nrow,ncol,i,inbas,tempf)
  100 continue
      read(inbas,*)
      do 120 i=1,nlay
        call u2drel(tmp,tmp2,nrow,ncol,i,inbas,tempf)
  120 continue
      do 140 i=1,nper
        read(inbas,*)perlen(i)
  140 continue
c
c--- read appropriate MODFLOW solver input to obtain HCLOSE
      if(iunit(9).gt.0) then
        read(iunit(9),*)
        call sip5rp(2,mxiter,accl,hclose,djunk,
     &      iunit(9),ipcalc,iprsip,tempf,ifrefm)
      elseif(iunit(11).gt.0) then
        read(iunit(11),*)
        call sor5rp(mxiter,accl,hclose,iunit(11),
     &      iprsor,tempf,ifrefm)
      elseif(iunit(13).gt.0) then
        read(iunit(13),*)
        call pcg2rp(2,1,hclose,rclose,npcond,
     &      nbpol,relax,iprpcg,iunit(13),tempf,mutpcg,niter,djunk,damp)
      endif
c
c--- read LAYCON array to determine if MODFLOW model is unconfined  
	call bcf5al(isum,lenx,lcsc1,lchy,lcbot,lctop,lcsc2,lctrpy,
     &    iunit(1),iss,ncol,nrow,nlay,tempf,ibcfcb,lcwetd,iwdflg,lccvwd,
     &    wetfct,iwetit,ihdwet,hdry,iapart,ifrefm)
c
c--- rewind MODFLOW files for future use in current run - remove temp file
      rewind (unit=iunit(1))
C 9 IS SIP
      IF(IUNIT(9).GT.0)rewind (unit=iunit(9))
C 11 IS SOR
      IF(IUNIT(11).GT.0)rewind (unit=iunit(11))
C 13 IS PCG
      IF(IUNIT(13).GT.0)rewind (unit=iunit(13))
      rewind (unit=modnme)
      rewind (unit=inbas)
      close (unit=tempf,status='DELETE')
	DEALLOCATE (TMP)
c
      return
  900 call USTOP('MODFLOW Names File Not Found')
      end
c
c
      subroutine calmod(x,ldxm1,ilstfil,nprt,
     1	itnonl,MFCNVRG,ipert,LASTSIM)

c  main program from MODFLOW-96 modified to be called as a 
c  subroutine from MODOFC 
c
c  all modifications to the original MODFLOW main program are
c  in lower case characters
c
	USE OFCMOD, ONLY:  tempf,modnme
	USE GWM1BAS1, ONLY: GWMOUT
      USE GWM1HDC1, ONLY: GWM1HDC1OS 
      USE GWM1STC1, ONLY: GWM1STC1OS 
c
c  Input:
c    x - work space for modflow
c  Output:
c  Printing variables
c    ilstfil - the unit number for MODFLOW listing output 
c    nprt - a set of flags dictating the output
c
c  Subroutine History
c     original version to interact with MODFLOW by C.Sawyer, Dec 1990
c     modified by D. Ahlfeld, Oct. 1991 
c     updated by G. Riefler, July 1997 to work with MODLFOW-96
c     modified by D. Ahlfeld, July 2001 to grab streamflow output
c
C     ******************************************************************
C     MAIN CODE FOR U.S. GEOLOGICAL SURVEY MODULAR MODEL -- MODFLOW-96
C           BY MICHAEL G. MCDONALD AND ARLEN W. HARBAUGH
C     MODFLOW-88 documented in:
C           McDonald, M.G. and Harbaugh, A.W., 1988, A modular
C           three-dimensional finite-difference ground-water flow
C           model: U.S. Geological Survey Techniques of Water
C           Resources Investigations, Book 6, Chapter A1, 586 p.
C     MODFLOW-96 documented in:
C           Harbaugh, A.W. and McDonald, M.G., 1996, User's
C           documentation for the U.S. Geological Survey modular
C           finite-difference ground-water flow model: U.S. Geological
C           Survey Open-File Report 96-485
C-----VERSION 0950 23MAY1996 MAIN
C-----VERSION 1401 03DEC1996 -- added PCG2, STR1, IBS1, CHD1, GFD1,
C                               HFB1, TLK1, DE45, and RES1 as documented
C                               in USGS reports
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
c
c--modofc-all real variables in MODFLOW are redeclared as double precision
      implicit double precision (a-h,o-z)
	LOGICAL MFCNVRG, LASTSIM
c
c--modofc-MODFLOW primary storage array is passed from MODOFC
      dimension x(ldxm1)
c-routines GWM1HDC1OS and GWM1STC1OS both require changes in data types
	REAL HDRYS
	parameter (mxstrmp=10000)  ! assume this is the maximum size
      real    strms(11,mxstrmp)
	integer istrms(5,mxstrmp)
c
c--modofc-dimension MODOFC variable passed through calling statement
	dimension nprt(3)
C1------SPECIFY THE SIZE OF THE X ARRAY.  TO CHANGE THE SIZE OF THE
C1------X ARRAY, CHANGE VALUE OF LENX IN THE NEXT STATEMENT.
c     PARAMETER (LENX=1500000)
c     COMMON X(LENX)
      COMMON /FLWCOM/LAYCON(200)
c
      CHARACTER*16 VBNM(40)
      CHARACTER*80 HEADNG(2)
      DIMENSION VBVL(4,40),IUNIT(40)
c     DOUBLE PRECISION DUMMY
c     EQUIVALENCE (DUMMY,X(1))
      CHARACTER*20 CHEDFM,CDDNFM
c     CHARACTER*80 FNAME
c     LOGICAL EXISTS
      CHARACTER*4 CUNIT(40)
      DATA CUNIT/'BCF ','WEL ','DRN ','RIV ','EVT ','TLK ','GHB ',
     1           'RCH ','SIP ','DE4 ','SOR ','OC  ','PCG ','GFD ',
     2           '    ','HFB ','RES ','STR ','IBS ','CHD ','    ',
     3           '    ','    ','    ','    ','    ','    ','    ',
     4           '    ','    ','    ','    ','    ','    ','    ',
     5           '    ','    ','    ','    ','    '/
C     ------------------------------------------------------------------
c
c--modofc-set the dimension of MODFLOW X array based on available storage
      lenx = ldxm1
c
c--modofc-open temporary file for writing MODFLOW listing
      open(unit=tempf,file='temp')
c
c--modofc-write message to iteration file - message will be erased if
c         MODFLOW successfully completes this run
   60 write(GWMOUT,*)'MODFLOW has crashed'
c
c--modofc-rewind all MODFLOW input files for the base run in perturubation
c         loop and the final run with optimal pumping
c      if(LASTSIM .or. IPERT.EQ.0) then
        rewind(unit=modnme)
	  rewind(unit=inbas)
	  do 65 i=1,40
	    if(iunit(i).gt.0) rewind(unit=iunit(i))
   65	  continue
c      endif
c
c--modofc-read in main problem on first and last run
      if(LASTSIM .or. IPERT.LE.0)then
c
c--modofc-setting iout to 0 will cause MODFLOW to reset it to listing file
        iout=0
c
c--modofc-setting iout to tempf will cause MODFLOW listing to go to tempf
        if(nprt(1).eq.0) iout=tempf
c
c--modofc-set the MODFLOW names file unit number to value set by MODOFC
      INUNIT=modnme
c
c--modofc-if not the first or last call to calmod
      else
c
c--modofc-reset MODFLOW time counter
        totim = 0.0
c
c--modofc-reinitialize heads and storage 
	  if(iss.eq.0) then
          call reset(x(LCHNEW),x(lcstrt),nodes,vbvl)
	  else
	    call reset(x(lchnew),x(lchold),nodes,vbvl)
	  endif
      endif
c
c--modofc-this is commented out because it is now done in optin
c
c     IBUNIT=98
c     IBOUTS=97
c     IBATCH=0
c     INQUIRE(FILE='modflow.bf',EXIST=EXISTS)
c     IF(EXISTS) THEN
c        IBATCH=1
c        OPEN(UNIT=IBUNIT,FILE='modflow.bf',STATUS='OLD')
c        OPEN(UNIT=IBOUTS,FILE='modbatch.rpt')
c        WRITE(IBOUTS,*) ' USGS MODFLOW MODEL BATCH-MODE REPORT'
c     END IF
C
C2------OPEN FILE OF FILE NAMES.
c50   IF(IBATCH.GT.0) THEN
c        READ(IBUNIT,'(A)',END=500) FNAME
c        IF(FNAME.EQ.' ') GO TO 50
c        WRITE(IBOUTS,'(1X,/1X,A)') FNAME
c     ELSE
c        WRITE(*,*) ' Enter the name of the NAME FILE:'
c        READ(*,'(A)') FNAME
c     END IF
c     INQUIRE(FILE=FNAME,EXIST=EXISTS)
c     IF(.NOT.EXISTS) THEN
c        IF(IBATCH.GT.0) THEN
c           WRITE(IBOUTS,*) ' Specified name file does not exist.'
c           WRITE(IBOUTS,*) ' Processing will continue with the next ',
c    1                      'name file in modflow.bf.'
c        ELSE
c           WRITE(*,*) ' File does not exist'
c        END IF
c        GO TO 50
c     END IF
c     OPEN(UNIT=INUNIT,FILE=FNAME,STATUS='OLD')
C
C3------DEFINE PROBLEM--ROWS,COLUMNS,LAYERS,STRESS PERIODS,PACKAGES.
      CALL BAS5DF(ISUM,HEADNG,NPER,ITMUNI,TOTIM,NCOL,NROW,NLAY,
     1        NODES,INBAS,IOUT,IUNIT,CUNIT,INUNIT,IXSEC,ICHFLG,IFREFM,
     2        ilstfil)
C
C4------ALLOCATE SPACE IN "X" ARRAY.
      CALL BAS5AL(ISUM,LENX,LCHNEW,LCHOLD,LCIBOU,LCCR,LCCC,LCCV,
     1              LCHCOF,LCRHS,LCDELR,LCDELC,LCSTRT,LCBUFF,LCIOFL,
     2              INBAS,ISTRT,NCOL,NROW,NLAY,IOUT,IAPART,IFREFM)
      IF(IUNIT(1).GT.0) CALL BCF5AL(ISUM,LENX,LCSC1,LCHY,
     1     LCBOT,LCTOP,LCSC2,LCTRPY,IUNIT(1),ISS,
     2     NCOL,NROW,NLAY,IOUT,IBCFCB,LCWETD,IWDFLG,LCCVWD,
     3     WETFCT,IWETIT,IHDWET,HDRY,IAPART,IFREFM)
      IF(IUNIT(2).GT.0) CALL WEL5AL(ISUM,LENX,LCWELL,MXWELL,NWELLS,
     1                 IUNIT(2),IOUT,IWELCB,NWELVL,IWELAL,IFREFM)
      IF(IUNIT(3).GT.0) CALL DRN5AL(ISUM,LENX,LCDRAI,NDRAIN,MXDRN,
     1                 IUNIT(3),IOUT,IDRNCB,NDRNVL,IDRNAL,IFREFM)
      IF(IUNIT(4).GT.0) CALL RIV5AL(ISUM,LENX,LCRIVR,MXRIVR,NRIVER,
     1            IUNIT(4),IOUT,IRIVCB,NRIVVL,IRIVAL,IFREFM)
      IF(IUNIT(5).GT.0) CALL EVT5AL(ISUM,LENX,LCIEVT,LCEVTR,LCEXDP,
     1            LCSURF,NCOL,NROW,NEVTOP,IUNIT(5),IOUT,IEVTCB,IFREFM)
      IF(IUNIT(6).GT.0) CALL TLK1AL(ISUM,LENX,NCOL,NROW,NLAY,
     1          LCRAT,LCZCB,LCA1,LCB1,LCALPH,LCBET,LCRM1,LCRM2,LCRM3,
     2          LCRM4,LCTL,LCTLK,LCSLU,LCSLD,NODES1,NM1,NM2,NUMC,
     3          NTM1,ITLKSV,ITLKRS,ITLKCB,ISS,IUNIT(6),IOUT)
      IF(IUNIT(7).GT.0) CALL GHB5AL(ISUM,LENX,LCBNDS,NBOUND,MXBND,
     1            IUNIT(7),IOUT,IGHBCB,NGHBVL,IGHBAL,IFREFM)
      IF(IUNIT(8).GT.0) CALL RCH5AL(ISUM,LENX,LCIRCH,LCRECH,NRCHOP,
     1            NCOL,NROW,IUNIT(8),IOUT,IRCHCB,IFREFM)
      IF(IUNIT(9).GT.0) CALL SIP5AL(ISUM,LENX,LCEL,LCFL,LCGL,LCV,
     1          LCHDCG,LCLRCH,LCW,MXITER,NPARM,NCOL,NROW,NLAY,
     2          IUNIT(9),IOUT,IFREFM)
      IF(IUNIT(10).GT.0) CALL DE45AL(ISUM,LENX,LCAU,LCAL,LCIUPP,
     1           LCIEQP,LCD4B,LCLRCH,LCHDCG,
     2           MXUP,MXLOW,MXEQ,MXBW,IUNIT(10),ITMX,ID4DIR,
     3           NCOL,NROW,NLAY,IOUT,ID4DIM)
      IF(IUNIT(11).GT.0) CALL SOR5AL(ISUM,LENX,LCA,LCRES,LCHDCG,LCLRCH,
     1       LCIEQP,MXITER,NCOL,NLAY,NSLICE,MBW,IUNIT(11),IOUT,IFREFM)
      IF(IUNIT(13).GT.0) CALL PCG2AL(ISUM,LENX,LCV,LCSS,LCP,LCCD,
     1       LCHCHG,LCLHCH,LCRCHG,LCLRCH,MXITER,ITER1,NCOL,NROW,NLAY,
     2       IUNIT(13),IOUT,NPCOND,LCIT1)
      IF(IUNIT(14).GT.0) CALL GFD1AL(ISUM,LENX,LCSC1,LCCDTR,LCCDTC,
     1     LCBOT,LCTOP,LCSC2,IUNIT(14),ISS,NCOL,NROW,NLAY,IOUT,IGFDCB)
      IF(IUNIT(16).GT.0) CALL HFB1AL(ISUM,LENX,LCHFBR,NHFB,IUNIT(16),      
     1           IOUT)                                                     
      IF(IUNIT(17).GT.0) CALL RES1AL(ISUM,LENX,LCIRES,LCIRSL,LCBRES,
     1  LCCRES,LCBBRE,LCHRES,LCHRSE,IUNIT(17),IOUT,NRES,IRESCB,
     2  NRESOP,IRESPT,NPTS,NCOL,NROW)
      IF(IUNIT(18).GT.0) CALL STR1AL(ISUM,LENX,LCSTRM,ICSTRM,MXSTRM,    
     1                 NSTREM,IUNIT(18),IOUT,ISTCB1,ISTCB2,NSS,NTRIB,   
     2                  NDIV,ICALC,CONST,LCTBAR,LCTRIB,LCIVAR,LCFGAR)   
      IF (IUNIT(19).GT.0) CALL IBS1AL(ISUM,LENX,LCHC,LCSCE,LCSCV,       
     1           LCSUB,NCOL,NROW,NLAY,IIBSCB,IIBSOC,ISS,IUNIT(19),IOUT) 
      IF(IUNIT(20).GT.0) CALL CHD1AL(ISUM,LENX,LCCHDS,NCHDS,MXCHD,      
     1           IUNIT(20),IOUT)                                        
C
C5------IF THE "X" ARRAY IS NOT BIG ENOUGH THEN STOP.
      IF(ISUM-1.GT.LENX) STOP
C
C6------READ AND PREPARE INFORMATION FOR ENTIRE SIMULATION.
      CALL BAS5RP(X(LCIBOU),X(LCHNEW),X(LCSTRT),X(LCHOLD),
     1       ISTRT,INBAS,HEADNG,NCOL,NROW,NLAY,VBVL,X(LCIOFL),
     2       IUNIT(12),IHEDFM,IDDNFM,IHEDUN,IDDNUN,IOUT,IPEROC,ITSOC,
     3       CHEDFM,CDDNFM,IBDOPT,IXSEC,LBHDSV,LBDDSV,IFREFM)
      IF(IUNIT(1).GT.0) CALL BCF5RP(X(LCIBOU),X(LCHNEW),X(LCSC1),
     1          X(LCHY),X(LCCR),X(LCCC),X(LCCV),X(LCDELR),
     2     X(LCDELC),X(LCBOT),X(LCTOP),X(LCSC2),X(LCTRPY),IUNIT(1),
     3     ISS,NCOL,NROW,NLAY,IOUT,X(LCWETD),IWDFLG,X(LCCVWD))
      IF(IUNIT(6).GT.0) CALL TLK1RP(X(LCRAT),X(LCZCB),X(LCA1),X(LCB1),
     1          X(LCALPH),X(LCBET),X(LCRM1),X(LCRM2),X(LCRM3),X(LCRM4),
     2          NODES1,NM1,NM2,NUMC,NTM1,ITLKRS,DELTM1,X(LCBUFF),
     3          X(LCDELC),X(LCDELR),TLKTIM,NROW,NCOL,IUNIT(6),IOUT)
      IF(IUNIT(9).GT.0) CALL SIP5RP(NPARM,MXITER,ACCL,HCLOSE,X(LCW),
     1          IUNIT(9),IPCALC,IPRSIP,IOUT,IFREFM)
      IF(IUNIT(10).GT.0) CALL DE45RP(IUNIT(10),MXITER,NITER,ITMX,
     1            ACCL,HCLOSE,IFREQ,IPRD4,IOUT,MUTD4)
      IF(IUNIT(11).GT.0) CALL SOR5RP(MXITER,ACCL,HCLOSE,IUNIT(11),
     1         IPRSOR,IOUT,IFREFM)
      IF(IUNIT(13).GT.0) CALL PCG2RP(MXITER,ITER1,HCLOSE,RCLOSE,
     1         NPCOND,NBPOL,RELAX,IPRPCG,IUNIT(13),IOUT,MUTPCG,
     2         NITER,X(LCIT1),DAMP)
      IF(IUNIT(14).GT.0) CALL GFD1RP(X(LCIBOU),X(LCHNEW),X(LCSC1),
     1          X(LCCDTR),X(LCCDTC),X(LCCR),X(LCCC),X(LCCV),X(LCDELR),
     2          X(LCDELC),X(LCBOT),X(LCTOP),X(LCSC2),
     3          IUNIT(14),ISS,NCOL,NROW,NLAY,NODES,IOUT)
      IF(IUNIT(16).GT.0) CALL HFB1RP(X(LCCR),X(LCCC),X(LCDELR),            
     1         X(LCDELC),X(LCHFBR),IUNIT(16),NCOL,NROW,NLAY,NODES,         
     1         NHFB,IOUT)                                               
      IF(IUNIT(19).GT.0) CALL IBS1RP(X(LCDELR),X(LCDELC),X(LCHNEW),     
     1      X(LCHC),X(LCSCE),X(LCSCV),X(LCSUB),NCOL,NROW,NLAY,          
     2      NODES,IIBSOC,ISUBFM,ICOMFM,IHCFM,ISUBUN,ICOMUN,IHCUN,       
     3      IUNIT(19),IOUT)                                             
c
c--modofc-if not the first or last call to calmod
c      else
c
c--modofc-reset MODFLOW time counter
c        totim = 0.0
c
c--modofc-reinitialize heads and storage 
c	  if(iss.eq.0) then
c          call reset(x(LCHNEW),x(lcstrt),nodes,vbvl)c
c	  else
c	    call reset(x(lchnew),x(lchold),nodes,vbvl)
c	  endif
c
c--modofc-rewind stress period information
c        do 70 i=1,nper
c
c--modofc-backspace in the BAS file to the start of stress period info
c	    backspace(unit=inbas)
c   70	  continue
c
c--modofc-for WEL, DRN, RIV, EVT, GHB, RCH, RES, STR and CHD packages
c       rewind input file and re-read past the first line of input file
c        do 80 i=1,9
c	    if(i.eq.1) j=2
c	    if(i.eq.2) j=3
c	    if(i.eq.3) j=4
c	    if(i.eq.4) j=5
c	    if(i.eq.5) j=7
c	    if(i.eq.6) j=8
c	    if(i.eq.7) j=17
c	    if(i.eq.8) j=18
c	    if(i.eq.9) j=20
c          if(iunit(j).gt.0) then
c	      rewind(unit=iunit(j))c
c	      read(iunit(j),*)
c          endif
c   80   continue
c
c--modofc-reset output control
c	  if(iunit(12).gt.0) rewind (unit=iunit(12))
c        call sbas5i(nlay,istrt,x(lciofl),iunit(12),iout,ihedfm,
c     &    iddnfm,ihedun,iddnun,iperoc,itsoc,chedfm,cddnfm,ibdopt,
c     &    lbhdsv,lbddsv,ifrerm)
c      endif
C
C7------SIMULATE EACH STRESS PERIOD.
      DO 300 KPER=1,NPER
      KKPER=KPER
C
C7A-----READ STRESS PERIOD TIMING INFORMATION.
      CALL BAS5ST(NSTP,DELT,TSMULT,PERTIM,KKPER,INBAS,IOUT,IFREFM)
C
C7B-----READ AND PREPARE INFORMATION FOR STRESS PERIOD.
      IF(IUNIT(2).GT.0) CALL WEL5RP(X(LCWELL),NWELLS,MXWELL,IUNIT(2),
     1             IOUT,NWELVL,IWELAL,IFREFM)
      IF(IUNIT(3).GT.0) CALL DRN5RP(X(LCDRAI),NDRAIN,MXDRN,IUNIT(3),
     1                 IOUT,NDRNVL,IDRNAL,IFREFM)
      IF(IUNIT(4).GT.0) CALL RIV5RP(X(LCRIVR),NRIVER,MXRIVR,IUNIT(4),
     1            IOUT,NRIVVL,IRIVAL,IFREFM)
      IF(IUNIT(5).GT.0) CALL EVT5RP(NEVTOP,X(LCIEVT),X(LCEVTR),
     1            X(LCEXDP),X(LCSURF),X(LCDELR),X(LCDELC),NCOL,NROW,
     1            IUNIT(5),IOUT,IFREFM)
      IF(IUNIT(7).GT.0) CALL GHB5RP(X(LCBNDS),NBOUND,MXBND,IUNIT(7),
     1              IOUT,NGHBVL,IGHBAL,IFREFM)
      IF(IUNIT(8).GT.0) CALL RCH5RP(NRCHOP,X(LCIRCH),X(LCRECH),
     1            X(LCDELR),X(LCDELC),NROW,NCOL,IUNIT(8),IOUT,IFREFM)
      IF(IUNIT(17).GT.0) CALL RES1RP(X(LCIRES),X(LCIRSL),X(LCBRES),
     1   X(LCCRES),X(LCBBRE),X(LCHRSE),X(LCIBOU),X(LCDELR),X(LCDELC),
     2   NRES,NRESOP,NPTS,NCOL,NROW,NLAY,PERLEN,DELT,NSTP,TSMULT,
     3   IUNIT(17),IOUT)
      IF(IUNIT(18).GT.0) CALL STR1RP(X(LCSTRM),X(ICSTRM),NSTREM,        
     1                     MXSTRM,IUNIT(18),IOUT,X(LCTBAR),NDIV,NSS,    
     2                     NTRIB,X(LCIVAR),ICALC,IPTFLG)                
      IF(IUNIT(20).GT.0) CALL CHD1RP(X(LCCHDS),NCHDS,MXCHD,X(LCIBOU),   
     1            NCOL,NROW,NLAY,PERLEN,DELT,NSTP,TSMULT,IUNIT(20),IOUT)
C
c--modofc-set locations and pump rates for candidate wells
      call GWF1GWM1RP(x(lcwell),nwells,nwelvl,mxwell,kper,nlay,iout)
c
c--modofc-select file for writing MODFLOW output to iout
c--       tempf is a temporary file which will be erased
      iout=tempf
      if(nprt(2).gt.0) then
c
c--modofc-if print flags are set print to ilistfil, MODFLOW listing file
	if(mod(itnonl,nprt(2)).eq.0) iout=ilstfil
      endif
C
C7C-----SIMULATE EACH TIME STEP.
      DO 200 KSTP=1,NSTP
      KKSTP=KSTP
C
C7C1----CALCULATE TIME STEP LENGTH. SET HOLD=HNEW..
      CALL BAS5AD(DELT,TSMULT,TOTIM,PERTIM,X(LCHNEW),X(LCHOLD),KKSTP,
     1             NCOL,NROW,NLAY)
      IF(IUNIT(6).GT.0) CALL TLK1AD(X(LCRAT),X(LCZCB),X(LCA1),X(LCB1),
     1          X(LCALPH),X(LCBET),X(LCRM1),X(LCRM2),X(LCRM3),X(LCRM4),
     2          X(LCTL),X(LCTLK),X(LCSLU),X(LCSLD),NM1,NM2,NUMC,NTM1,
     3          DELTM1,X(LCHNEW),X(LCIBOU),X(LCTOP),
     4          NROW,NCOL,NLAY,DELT,TLKTIM,IUNIT(6),IOUT)
      IF(IUNIT(20).GT.0) CALL CHD1FM(NCHDS,MXCHD,X(LCCHDS),X(LCIBOU),   
     1          X(LCHNEW),X(LCHOLD),PERLEN,PERTIM,DELT,NCOL,NROW,NLAY)  
      IF(IUNIT(1).GT.0) CALL BCF5AD(X(LCIBOU),X(LCHOLD),X(LCBOT),
     1             X(LCWETD),IWDFLG,ISS,NCOL,NROW,NLAY)
      IF(IUNIT(17).GT.0) CALL RES1AD(X(LCHRES),X(LCHRSE),X(LCIRES),
     1 X(LCBRES),X(LCDELR),X(LCDELC),NRES,IRESPT,NCOL,NROW,
     1      PERLEN,PERTIM,TOTIM,KKSTP,KKPER,IOUT)
C
C7C2----ITERATIVELY FORMULATE AND SOLVE THE EQUATIONS.
      DO 100 KITER=1,MXITER
      KKITER=KITER
C
C7C2A---FORMULATE THE FINITE DIFFERENCE EQUATIONS.
      CALL BAS5FM(X(LCHCOF),X(LCRHS),NODES)
      IF(IUNIT(1).GT.0) CALL BCF5FM(X(LCHCOF),X(LCRHS),X(LCHOLD),
     1          X(LCSC1),X(LCHNEW),X(LCIBOU),X(LCCR),X(LCCC),X(LCCV),
     2          X(LCHY),X(LCTRPY),X(LCBOT),X(LCTOP),X(LCSC2),
     3          X(LCDELR),X(LCDELC),DELT,ISS,KKITER,KKSTP,KKPER,NCOL,
     4          NROW,NLAY,IOUT,X(LCWETD),IWDFLG,X(LCCVWD),WETFCT,
     5          IWETIT,IHDWET,HDRY,X(LCBUFF))
      IF(IUNIT(14).GT.0) CALL GFD1FM(X(LCHCOF),X(LCRHS),X(LCHOLD),
     1          X(LCSC1),X(LCHNEW),X(LCIBOU),X(LCCR),X(LCCC),X(LCCV),
     2          X(LCCDTR),X(LCCDTC),X(LCBOT),X(LCTOP),X(LCSC2),
     3          DELT,ISS,KKITER,KKSTP,KKPER,NCOL,NROW,NLAY,IOUT)
      IF(IUNIT(16).GT.0) CALL HFB1FM(X(LCHNEW),X(LCCR),X(LCCC),            
     1          X(LCBOT),X(LCTOP),X(LCDELR),X(LCDELC),X(LCHFBR),           
     2          NCOL,NROW,NLAY,NHFB)                                       
      IF(IUNIT(6).GT.0) CALL TLK1FM(X(LCRAT),X(LCTL),X(LCTLK),X(LCSLU),
     1          X(LCSLD),NUMC,X(LCHNEW),X(LCIBOU),X(LCTOP),X(LCCV),
     2          X(LCHCOF),X(LCRHS),NROW,NCOL,NLAY)
      IF(IUNIT(2).GT.0) CALL WEL5FM(NWELLS,MXWELL,X(LCRHS),X(LCWELL),
     1           X(LCIBOU),NCOL,NROW,NLAY,NWELVL)
      IF(IUNIT(3).GT.0) CALL DRN5FM(NDRAIN,MXDRN,X(LCDRAI),X(LCHNEW),
     1         X(LCHCOF),X(LCRHS),X(LCIBOU),NCOL,NROW,NLAY,NDRNVL)
      IF(IUNIT(4).GT.0) CALL RIV5FM(NRIVER,MXRIVR,X(LCRIVR),X(LCHNEW),
     1            X(LCHCOF),X(LCRHS),X(LCIBOU),NCOL,NROW,NLAY,NRIVVL)
      IF(IUNIT(5).GT.0) CALL EVT5FM(NEVTOP,X(LCIEVT),X(LCEVTR),
     1            X(LCEXDP),X(LCSURF),X(LCRHS),X(LCHCOF),X(LCIBOU),
     1            X(LCHNEW),NCOL,NROW,NLAY)
      IF(IUNIT(7).GT.0) CALL GHB5FM(NBOUND,MXBND,X(LCBNDS),X(LCHCOF),
     1            X(LCRHS),X(LCIBOU),NCOL,NROW,NLAY,NGHBVL)
      IF(IUNIT(8).GT.0) CALL RCH5FM(NRCHOP,X(LCIRCH),X(LCRECH),
     1            X(LCRHS),X(LCIBOU),NCOL,NROW,NLAY)
      IF(IUNIT(17).GT.0) CALL RES1FM(X(LCIRES),X(LCIRSL),X(LCBRES),
     1   X(LCCRES),X(LCBBRE),X(LCHRES),X(LCIBOU),X(LCHNEW),X(LCHCOF),
     2   X(LCRHS),NRES,NRESOP,NCOL,NROW,NLAY)
      IF(IUNIT(18).GT.0) CALL STR1FM(NSTREM,X(LCSTRM),X(ICSTRM),        
     1                     X(LCHNEW),X(LCHCOF),X(LCRHS),X(LCIBOU),      
     2              MXSTRM,NCOL,NROW,NLAY,IOUT,NSS,X(LCTBAR),           
     3              NTRIB,X(LCTRIB),X(LCIVAR),X(LCFGAR),ICALC,CONST)    
      IF(IUNIT(19).GT.0) CALL IBS1FM(X(LCRHS),X(LCHCOF),X(LCHNEW),      
     1       X(LCHOLD),X(LCHC),X(LCSCE),X(LCSCV),X(LCIBOU),             
     2       NCOL,NROW,NLAY,DELT)                                       
C
C7C2B---MAKE ONE CUT AT AN APPROXIMATE SOLUTION.
      IF(IUNIT(9).GT.0) CALL SIP5AP(X(LCHNEW),X(LCIBOU),X(LCCR),X(LCCC),
     1     X(LCCV),X(LCHCOF),X(LCRHS),X(LCEL),X(LCFL),X(LCGL),X(LCV),
     2     X(LCW),X(LCHDCG),X(LCLRCH),NPARM,KKITER,HCLOSE,ACCL,ICNVG,
     3     KKSTP,KKPER,IPCALC,IPRSIP,MXITER,NSTP,NCOL,NROW,NLAY,NODES,
     4     IOUT)
      IF(IUNIT(10).GT.0) CALL DE45AP(X(LCHNEW),X(LCIBOU),X(LCAU),
     1  X(LCAL),X(LCIUPP),X(LCIEQP),X(LCD4B),MXUP,MXLOW,MXEQ,MXBW,
     2  X(LCCR),X(LCCC),X(LCCV),X(LCHCOF),X(LCRHS),ACCL,KKITER,ITMX,
     3  MXITER,NITER,HCLOSE,IPRD4,ICNVG,NCOL,NROW,NLAY,IOUT,X(LCLRCH),
     4  X(LCHDCG),IFREQ,KKSTP,KKPER,DELT,NSTP,ID4DIR,ID4DIM,MUTD4)
      IF(IUNIT(11).GT.0) CALL SOR5AP(X(LCHNEW),X(LCIBOU),X(LCCR),
     1     X(LCCC),X(LCCV),X(LCHCOF),X(LCRHS),X(LCA),X(LCRES),X(LCIEQP),
     2     X(LCHDCG),X(LCLRCH),KKITER,HCLOSE,ACCL,ICNVG,KKSTP,KKPER,
     3     IPRSOR,MXITER,NSTP,NCOL,NROW,NLAY,NSLICE,MBW,IOUT)
      IF(IUNIT(13).GT.0) CALL PCG2AP(X(LCHNEW),X(LCIBOU),X(LCCR),
     1      X(LCCC),X(LCCV),X(LCHCOF),X(LCRHS),X(LCV),X(LCSS),X(LCP),
     2      X(LCCD),X(LCHCHG),X(LCLHCH),X(LCRCHG),X(LCLRCH),KKITER,
     3      NITER,HCLOSE,RCLOSE,ICNVG,KKSTP,KKPER,IPRPCG,MXITER,ITER1,
     4      NPCOND,NBPOL,NSTP,NCOL,NROW,NLAY,NODES,RELAX,IOUT,MUTPCG,
     5      0,0,SN,SP,SR,X(LCIT1),DAMP)
C
C7C2C---IF CONVERGENCE CRITERION HAS BEEN MET STOP ITERATING.
      IF(ICNVG.EQ.1) GO TO 110
  100 CONTINUE
      KITER=MXITER
  110 CONTINUE
c
c--modofc-select file for writing MODFLOW output to iout
c--       tempf is a temporary file which will be erased
      iout=tempf
      if(nprt(3).gt.0) then
c
c--modofc-if print flags are set print to ilistfil, MODFLOW listing file
        if(mod(itnonl,nprt(3)).eq.0) iout=ilstfil
      endif
C
C7C3----DETERMINE WHICH OUTPUT IS NEEDED.
      CALL BAS5OC(NSTP,KKSTP,ICNVG,X(LCIOFL),NLAY,IBUDFL,ICBCFL,
     1   IHDDFL,IUNIT(12),IOUT,KKPER,IPEROC,ITSOC,IBDOPT,IXSEC,IFREFM)
C
C7C4----CALCULATE BUDGET TERMS. SAVE CELL-BY-CELL FLOW TERMS.
      MSUM=1
      IF(IUNIT(6).GT.0) CALL TLK1BD(X(LCRAT),X(LCTL),X(LCTLK),
     1          X(LCSLU),X(LCSLD),NUMC,ITLKCB,X(LCHNEW),X(LCBUFF),
     2          X(LCIBOU),X(LCTOP),X(LCCV),VBNM,VBVL,MSUM,NCOL,NROW,
     3          NLAY,DELT,KSTP,KPER,ICBCFL,IOUT)
C7C4A---THE ORIGINAL BCF BUDGET MODULE HAS BEEN REPLACED BY THREE
C7C4A---SUBMODULES: SBCF5S, SBCF5F, AND SBCF5B .
      IF(IUNIT(1).GT.0) THEN
         CALL SBCF5S(VBNM,VBVL,MSUM,X(LCHNEW),X(LCIBOU),X(LCHOLD),
     1     X(LCSC1),X(LCTOP),X(LCSC2),DELT,ISS,NCOL,NROW,NLAY,KKSTP,
     2     KKPER,IBCFCB,ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM)
         CALL SBCF5F(VBNM,VBVL,MSUM,X(LCHNEW),X(LCIBOU),X(LCCR),
     1     X(LCCC),X(LCCV),X(LCTOP),DELT,NCOL,NROW,NLAY,KKSTP,KKPER,
     2     IBCFCB,X(LCBUFF),IOUT,ICBCFL,PERTIM,TOTIM,ICHFLG)
         IBDRET=0
         IC1=1
         IC2=NCOL
         IR1=1
         IR2=NROW
         IL1=1
         IL2=NLAY
         DO 155 IDIR=1,3
         CALL SBCF5B(X(LCHNEW),X(LCIBOU),X(LCCR),X(LCCC),X(LCCV),
     1      X(LCTOP),NCOL,NROW,NLAY,KKSTP,KKPER,IBCFCB,X(LCBUFF),
     2      IOUT,ICBCFL,DELT,PERTIM,TOTIM,IDIR,IBDRET,ICHFLG,
     3      IC1,IC2,IR1,IR2,IL1,IL2)
155      CONTINUE
      END IF
      IF(IUNIT(14).GT.0) CALL GFD1BD(VBNM,VBVL,MSUM,X(LCHNEW),
     1     X(LCIBOU),X(LCHOLD),X(LCSC1),X(LCCR),X(LCCC),X(LCCV),
     2     X(LCTOP),X(LCSC2),DELT,ISS,NCOL,NROW,NLAY,KKSTP,KKPER,
     3     IGFDCB,ICBCFL,X(LCBUFF),IOUT)
      IF(IUNIT(2).GT.0) CALL WEL5BD(NWELLS,MXWELL,VBNM,VBVL,MSUM,
     1     X(LCWELL),X(LCIBOU),DELT,NCOL,NROW,NLAY,KKSTP,KKPER,IWELCB,
     1     ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM,NWELVL,IWELAL)
      IF(IUNIT(3).GT.0) CALL DRN5BD(NDRAIN,MXDRN,VBNM,VBVL,MSUM,
     1     X(LCDRAI),DELT,X(LCHNEW),NCOL,NROW,NLAY,X(LCIBOU),KKSTP,
     2     KKPER,IDRNCB,ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM,NDRNVL,
     3     IDRNAL)
      IF(IUNIT(4).GT.0) CALL RIV5BD(NRIVER,MXRIVR,X(LCRIVR),X(LCIBOU),
     1     X(LCHNEW),NCOL,NROW,NLAY,DELT,VBVL,VBNM,MSUM,KKSTP,KKPER,
     2     IRIVCB,ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM,NRIVVL,IRIVAL)
      IF(IUNIT(5).GT.0) CALL EVT5BD(NEVTOP,X(LCIEVT),X(LCEVTR),
     1     X(LCEXDP),X(LCSURF),X(LCIBOU),X(LCHNEW),NCOL,NROW,NLAY,
     2     DELT,VBVL,VBNM,MSUM,KKSTP,KKPER,IEVTCB,ICBCFL,X(LCBUFF),IOUT,
     3     PERTIM,TOTIM)
      IF(IUNIT(7).GT.0) CALL GHB5BD(NBOUND,MXBND,VBNM,VBVL,MSUM,
     1     X(LCBNDS),DELT,X(LCHNEW),NCOL,NROW,NLAY,X(LCIBOU),KKSTP,
     2     KKPER,IGHBCB,ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM,NGHBVL,
     3     IGHBAL)
      IF(IUNIT(8).GT.0) CALL RCH5BD(NRCHOP,X(LCIRCH),X(LCRECH),
     1     X(LCIBOU),NROW,NCOL,NLAY,DELT,VBVL,VBNM,MSUM,KKSTP,KKPER,
     2     IRCHCB,ICBCFL,X(LCBUFF),IOUT,PERTIM,TOTIM)
      IF(IUNIT(17).GT.0) CALL RES1BD(X(LCIRES),X(LCIRSL),X(LCBRES),
     1      X(LCCRES),X(LCBBRE),X(LCHRES),X(LCIBOU),X(LCHNEW),
     2      X(LCBUFF),VBVL,VBNM,MSUM,KSTP,KPER,NRES,NRESOP,
     3      NCOL,NROW,NLAY,DELT,IRESCB,ICBCFL,IOUT)
      IF(IUNIT(18).GT.0) CALL STR1BD(NSTREM,X(LCSTRM),X(ICSTRM),        
     1   X(LCIBOU),MXSTRM,X(LCHNEW),NCOL,NROW,NLAY,DELT,VBVL,VBNM,MSUM, 
     2   KKSTP,KKPER,ISTCB1,ISTCB2,ICBCFL,X(LCBUFF),IOUT,NTRIB,NSS,     
     3   X(LCTRIB),X(LCTBAR),X(LCIVAR),X(LCFGAR),ICALC,CONST,IPTFLG)    
      IF(IUNIT(19).GT.0) CALL IBS1BD(X(LCIBOU),X(LCHNEW),X(LCHOLD),     
     1      X(LCHC),X(LCSCE),X(LCSCV),X(LCSUB),X(LCDELR),X(LCDELC),     
     2      NCOL,NROW,NLAY,DELT,VBVL,VBNM,MSUM,KSTP,KPER,IIBSCB,        
     3      ICBCFL,X(LCBUFF),IOUT)                                      
C
C7C5---PRINT AND OR SAVE HEADS AND DRAWDOWNS. PRINT OVERALL BUDGET.
      CALL BAS5OT(X(LCHNEW),X(LCSTRT),ISTRT,X(LCBUFF),X(LCIOFL),
     1     MSUM,X(LCIBOU),VBNM,VBVL,KKSTP,KKPER,DELT,PERTIM,TOTIM,
     2     ITMUNI,NCOL,NROW,NLAY,ICNVG,IHDDFL,IBUDFL,IHEDFM,IHEDUN,
     3     IDDNFM,IDDNUN,IOUT,CHEDFM,CDDNFM,IXSEC,LBHDSV,LBDDSV,
c
     4     pdiffv,pdiffr)
c
c--modofc-check flow balance
       backspace (unit=GWMOUT)
       if(pdiffv.gt.1.or.pdiffr.gt.1) write(GWMOUT,120) pdiffv,pdiffr
  120  format(/'WARNING - the MODFLOW mass balance is >1%',
     &   '  volumetric percent difference = ',f5.1,'%'/,
     &   '  flow rate percent difference  = ',f5.2,'%')
       write(GWMOUT,*)'MODFLOW has crashed'
C
C7C5A--PRINT AND OR SAVE SUBSIDENCE, COMPACTION, AND CRITICAL HEAD.
      IF(IUNIT(19).GT.0) CALL IBS1OT(NCOL,NROW,NLAY,PERTIM,TOTIM,KSTP,  
     1      KPER,NSTP,X(LCBUFF),X(LCSUB),X(LCHC),IIBSOC,ISUBFM,ICOMFM,  
     2      IHCFM,ISUBUN,ICOMUN,IHCUN,IUNIT(19),IOUT)                   
C
C7C6----IF ITERATION FAILED TO CONVERGE THEN STOP.
c
c--modofc-write failure statement to file before stoping
c     IF(ICNVG.EQ.0) STOP
      if(icnvg.eq.0) then
        backspace (unit=GWMOUT)
        write(GWMOUT,*)'MODFLOW did not converge'
	  MFCNVRG = .FALSE.
        RETURN
      endif
  200 CONTINUE
c
c--modofc-assign system state to state array at end of stress period
      HDRYS=HDRY     ! this converts to single precision
      call GWM1HDC1OS(x(lchnew),ncol,nrow,nlay,kper,HDRYS)
      CALL SETINIT(x(lchnew),x(lchold),ncol,nrow,nlay,ipert)
c
c-----if stream is available then call
	if(nstrem.gt.0)then
        if(mxstrm.gt.mxstrmp)then
	    write(iout,*)'PROGRAM STOPPED: INCREASE DIMENSION OF MXSTRMP'
          stop
        endif
        call redimstr(X(LCstrm),X(ICstrm),mxstrm,strms,istrms,mxstrmp)
        call GWM1STC1OS(nstrem,strms,istrms,mxstrm,kper)
	endif
c
  300 CONTINUE
C
C7C7----WRITE RESTART RECORDS
C7C7A---WRITE RESTART RECORDS FOR TRANSIENT-LEAKAGE PACKAGE
      IF(IUNIT(6).GT.0) CALL TLK1OT(X(LCRM1),X(LCRM2),
     1     X(LCRM3),X(LCRM4),NM1,NM2,ITLKSV,DELTM1,TLKTIM,IOUT)
c
c--modofc-erase the MODFLOW listing output for this run
      close(unit=tempf,status='DELETE')
c
c--modofc-overwrite the line in the iteration file which says 'MODFLOW has crashed'
      backspace (unit=GWMOUT)
      write(GWMOUT,*)
      backspace (unit=GWMOUT)
C
C8------END OF SIMULATION
c
c--modofc-replace termination code with a return to calling program
	MFCNVRG = .TRUE.
      return
c     IF(IBATCH.GT.0) THEN
c        WRITE(IBOUTS,*) ' Normal termination of simulation.'
c        DO 400 I=1,IBOUTS-1
c           INQUIRE(UNIT=I,OPENED=EXISTS)
c           IF(EXISTS) CLOSE(I)
c 400    CONTINUE
c        GO TO 50
c     END IF
c 500 STOP
C
      END
c
c******************************************************************
c
      subroutine redimstr(strm,istrm,mxstrm,strms,istrms,mxstrmp)
c
c******************************************************************
c
c  Purpose: assign strm and istrm to separate arrays for use by GWM routines
c
c  Input
c     strm, istrm - array embedded in MF96 X array
c  Output
c     strms, istrms - data stored in their own arrays
c-----------------------------------------------------------------------
      double precision strm(11,mxstrm)
      real    strms(11,mxstrmp)
	integer istrms(5,mxstrmp),istrm(5,mxstrm)

        DO 180 is=1,mxstrm
          do 176 is1=1,11
	      ixloc = ixloc+1
            strms(is1,is)=strm(is1,is)    ! move strm to its own array
  176     enddo
          do 178 is1=1,5
            istrms(is1,is)=istrm(is1,is)  ! move istrm to its own array 
  178     enddo
  180   enddo

      return
      end
c
c******************************************************************
c
      subroutine SETINIT(hnew,hold,ncol,nrow,nlay,ipert)
c
c******************************************************************
c
c  Purpose: assign computed system state to MODOFC constraint state array
c
c  Input
c     hnew    - computed head values
c     nrow, ncol, nlay  - dimensioning indices
c-----------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      dimension hnew(ncol,nrow,nlay)
      dimension hold(ncol,nrow,nlay)
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
c--- on first call set the heads to be used for initial value of iter
        if(IPERT.LE.0)then
          do 9130 i=1,ncol
            do 9130 j=1,nrow
              do 9130 k=1,nlay
                hold(i,j,k) = hnew(i,j,k)
 9130      continue
        endif
      return
      end
c
c******************************************************************
c
      subroutine reset(hnew,a,nodes,vbvl)
c
c******************************************************************
c
c  Purpose: reset certain variables
c
c-----------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      dimension hnew(nodes),a(nodes),vbvl(4,40)
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
c--- set back initial heads
      do 100 i=1,nodes
        hnew(i)=a(i)
  100 continue
c--- set storage array for volumetric budget to zero
      do 110 i=1,40
        do 110 j=1,4
          vbvl(j,i) = 0.0d0
  110 continue
c
      return
      enD

C******************************************************************
      subroutine GWF1GWM1RP(well,nwells,nwelvl,mxwell,kper,nlay,iout)
c******************************************************************
c
c  Purpose: assign MODOFC candidate well locations to MODFLOW wells array
c
c  Input
c     FVILOC, FVJLOC, FVRATIO - location of MODOFC candidate well
c     nwelvl, mxwell, NFVAR, ldnpm, ldsp - dimensioning indices
c     FVSP - stress period indicator matrix
c     kper    - current stress period
c     iout    - unit number of error statements
c     nwells  - initial total number of wells
c  Output
c     well    - update MODFLOW well array
c     nwells  - updated total number of wells
c-----------------------------------------------------------------------
	USE GWM1DCV1, ONLY :  NFVAR,FVBASE,FVILOC,FVJLOC,
     1                      FVKLOC,FVNCELL,FVSP,FVRATIO
      implicit NONE
	INTEGER NWELLS,NWELVL,MXWELL,KPER,NLAY,IOUT
	DOUBLE PRECISION well(nwelvl,mxwell)
	INTEGER I,J,K
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
c--- calculate number of new wells to be added
	j = 0
	do 100 i=1,NFVAR
C-------IF PUMPING VARIABLE IS ACTIVE IN THIS STRESS PERIOD ADD ITS CELLS
	  if(FVSP(i,kper)) J = J + FVNCELL(I)
  100	continue
c     
c--- check that new wells do not exceed maximum space available
      if(nwells+j.gt.mxwell)then
        write(iout,120) mxwell,nwells+j
        stop
      endif
	j = nwells
C
C-----LOOP OVER PUMPING VARIABLES
      do 110 i=1,NFVAR
	  if(FVSP(i,kper)) then
C-------PUMPING VARIABLE IS active during this stress period
	    do 105 k=1,FVNCELL(I)
c
c--- transfer candidate well locations to MODFLOW array FOR EACH CELL
	        j = j + 1
              well(1,j)=dble(FVKLOC(I,K))
              well(2,j)=dble(FVILOC(i,K))
              well(3,j)=dble(FVJLOC(i,K))
              well(4,j)=FVBASE(i)*FVRATIO(i,k)
  105	    continue
	  endif
  110 continue
	nwells = j
c
      return
  120 format('MODOFC requires that mxwells in the WEL file be ',/,
     &       'equal to the sum of background and managed wells.',/,
     &    'Mxwells is currently',i4,' - increase it to',i4)
      end
      SUBROUTINE BAS5DF(ISUM,HEADNG,NPER,ITMUNI,TOTIM,NCOL,NROW,
     1    NLAY,NODES,INBAS,IOUT,IUNIT,CUNIT,INUNIT,IXSEC,ICHFLG,IFREFM,
     2    ilstfil)
C
C-----VERSION 1030 20FEB1996 BAS5DF
C     ******************************************************************
C     DEFINE KEY MODEL PARAMETERS
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*80 HEADNG(2)
      DIMENSION IUNIT(40)
      CHARACTER*4 CUNIT(40)
      CHARACTER*80 LINE1,LINE2
C     ------------------------------------------------------------------
C0------OPEN FILES AND ASSIGN IUNIT VALUES.
      CALL SBAS5O(INUNIT,INBAS,IOUT,IUNIT,CUNIT,ilstfil)
C
C1------PRINT THE NAME OF THE PROGRAM.
      WRITE(IOUT,1)
    1 FORMAT('1',33X,'MODFLOW',/6X,'U.S. GEOLOGICAL SURVEY MODULAR',
     1      ' FINITE-DIFFERENCE GROUND-WATER FLOW MODEL')
C
C2------READ AND PRINT A HEADING.
      READ(INBAS,'(A)') HEADNG(1)
      READ(INBAS,'(A)') HEADNG(2)
      WRITE(IOUT,'(1X,/1X,A)') HEADNG(1)
      WRITE(IOUT,'(1X,A)') HEADNG(2)
C
C3------READ LINE SPECIFYING NUMBER OF LAYERS,ROWS,COLUMNS,STRESS
C3------PERIODS AND UNITS OF TIME CODE, BUT DON'T DECODE UNTIL IT IS
C3------DETERMINED THAT FREE OR FIXED FORMAT IS BEING USED.
      READ(INBAS,'(A)') LINE1
C
C4------READ OPTIONS RECORD AND LOOK FOR OPTIONS
      READ(INBAS,'(A)') LINE2
      IXSEC=0
      ICHFLG=0
      IFREFM=0
      LLOC=1
    5 CALL URWORDd(LINE2,LLOC,ISTART,ISTOP,1,N,R,IOUT,INBAS)
      IF(LINE2(ISTART:ISTOP).EQ.'XSECTION') THEN
         IXSEC=1
      ELSE IF(LINE2(ISTART:ISTOP).EQ.'CHTOCH') THEN
         ICHFLG=1
      ELSE IF(LINE2(ISTART:ISTOP).EQ.'FREE') THEN
         IFREFM=1
         WRITE(IOUT,6)
    6    FORMAT (1X,'THE FREE FORMAT OPTION HAS BEEN SELECTED')
      END IF
      IF(LLOC.LT.80) GO TO 5
C
C5------READ NUMBER OF LAYERS, ROWS, COLUMNS, STRESS PERIODS, AND
C5------ITMUNI USING FREE OR FIXED FORMAT.
      IF(IFREFM.EQ.0) THEN
         READ(LINE1,'(5I10)') NLAY,NROW,NCOL,NPER,ITMUNI
      ELSE
         LLOC=1
         CALL URWORDd(LINE1,LLOC,ISTART,ISTOP,2,NLAY,R,IOUT,INBAS)
         CALL URWORDd(LINE1,LLOC,ISTART,ISTOP,2,NROW,R,IOUT,INBAS)
         CALL URWORDd(LINE1,LLOC,ISTART,ISTOP,2,NCOL,R,IOUT,INBAS)
         CALL URWORDd(LINE1,LLOC,ISTART,ISTOP,2,NPER,R,IOUT,INBAS)
         CALL URWORDd(LINE1,LLOC,ISTART,ISTOP,2,ITMUNI,R,IOUT,INBAS)
      END IF
C
C6------PRINT # OF LAYERS, ROWS, COLUMNS AND STRESS PERIODS.
      WRITE(IOUT,7) NLAY,NROW,NCOL
    7 FORMAT(1X,I4,' LAYERS',I10,' ROWS',I10,' COLUMNS')
      WRITE(IOUT,8) NPER
    8 FORMAT(1X,I3,' STRESS PERIOD(S) IN SIMULATION')
C
C7------SELECT AND PRINT A MESSAGE SHOWING TIME UNITS AND OTHER OPTIONS.
      IF(ITMUNI.LT.0 .OR. ITMUNI.GT.5) ITMUNI=0
      IF(ITMUNI.EQ.0) THEN
         WRITE(IOUT,9)
    9    FORMAT(1X,'MODEL TIME UNITS ARE UNDEFINED')
      ELSE IF(ITMUNI.EQ.1) THEN
         WRITE(IOUT,11)
   11    FORMAT(1X,'MODEL TIME UNIT IS SECONDS')
      ELSE IF(ITMUNI.EQ.2) THEN
         WRITE(IOUT,21)
   21    FORMAT(1X,'MODEL TIME UNIT IS MINUTES')
      ELSE IF(ITMUNI.EQ.3) THEN
         WRITE(IOUT,31)
   31    FORMAT(1X,'MODEL TIME UNIT IS HOURS')
      ELSE IF(ITMUNI.EQ.4) THEN
         WRITE(IOUT,41)
   41    FORMAT(1X,'MODEL TIME UNIT IS DAYS')
      ELSE
         WRITE(IOUT,51)
   51    FORMAT(1X,'MODEL TIME UNIT IS YEARS')
      END IF
      IF(IXSEC.NE.0) WRITE(IOUT,61)
   61 FORMAT(1X,'CROSS SECTION OPTION IS SPECIFIED')
      IF(ICHFLG.NE.0) WRITE(IOUT,62)
   62 FORMAT(1X,'CALCULATE FLOW BETWEEN ADJACENT CONSTANT-HEAD CELLS')
C
C8------INITIALIZE TOTAL ELAPSED TIME COUNTER STORAGE ARRAY COUNTER
C8------AND CALCULATE NUMBER OF CELLS.
      TOTIM=0.
      ISUM=1
      NODES=NCOL*NROW*NLAY
C
C9------RETURN
      RETURN
      END
      SUBROUTINE BAS5AL(ISUM,LENX,LCHNEW,LCHOLD,LCIBOU,LCCR,LCCC,LCCV,
     1            LCHCOF,LCRHS,LCDELR,LCDELC,LCSTRT,LCBUFF,LCIOFL,INBAS,
     2            ISTRT,NCOL,NROW,NLAY,IOUT,IAPART,IFREFM)
C-----VERSION 1334 20FEB1996 BAS5AL
C     ******************************************************************
C     ALLOCATE SPACE FOR BASIC MODEL ARRAYS
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
C     ------------------------------------------------------------------
C
C1------PRINT A MESSAGE IDENTIFYING THE PACKAGE.
      WRITE(IOUT,1)INBAS
    1 FORMAT(1X,/1X,'BAS5 -- BASIC MODEL PACKAGE, VERSION 5, 1/1/95',
     2' INPUT READ FROM UNIT',I3)
C
C2------READ & PRINT FLAG IAPART (RHS & BUFFER SHARE SPACE?) AND
C2------FLAG ISTRT (SHOULD STARTING HEADS BE KEPT FOR DRAWDOWN?).
      IF(IFREFM.EQ.0) THEN
         READ(INBAS,'(2I10)') IAPART,ISTRT
      ELSE
         READ(INBAS,*) IAPART,ISTRT
      END IF
      IF(IAPART.NE.0) WRITE(IOUT,2)
    2 FORMAT(1X,
     1    'ARRAYS RHS AND BUFF WILL HAVE SEPARATE MEMORY ALLOCATIONS')
      IF(IAPART.EQ.0) WRITE(IOUT,3)
    3 FORMAT(1X,'ARRAYS RHS AND BUFF WILL SHARE MEMORY')
      IF(ISTRT.NE.0) WRITE(IOUT,4)
    4 FORMAT(1X,'INITIAL HEAD WILL BE KEPT THROUGHOUT THE SIMULATION')
      IF(ISTRT.EQ.0) WRITE(IOUT,5)
    5 FORMAT(1X,'INITIAL HEAD WILL NOT BE KEPT THROUGHOUT THE',
     1 ' SIMULATION, WHICH MEANS',/1X,'DRAWDOWN CANNOT BE CALCULATED')
C
C3------STORE LOCATION OF FIRST UNALLOCATED SPACE IN X.
      ISOLD=ISUM
      NRCL=NROW*NCOL*NLAY
C
C4------ALLOCATE SPACE FOR ARRAYS.
      LCHNEW=ISUM
      ISUM=ISUM+NRCL
      LCHOLD=ISUM
      ISUM=ISUM+NRCL
      LCIBOU=ISUM
      ISUM=ISUM+NRCL
      LCCR=ISUM
      ISUM=ISUM+NRCL
      LCCC=ISUM
      ISUM=ISUM+NRCL
      LCCV=ISUM
      ISUM=ISUM+NROW*NCOL*(NLAY-1)
      LCHCOF=ISUM
      ISUM=ISUM+NRCL
      LCRHS=ISUM
      ISUM=ISUM+NRCL
      LCDELR=ISUM
      ISUM=ISUM+NCOL
      LCDELC=ISUM
      ISUM=ISUM+NROW
      LCIOFL=ISUM
      ISUM=ISUM+NLAY*4
C
C5------IF BUFFER AND RHS SHARE SPACE THEN LCBUFF=LCRHS.
      LCBUFF=LCRHS
      IF(IAPART.EQ.0) GO TO 50
      LCBUFF=ISUM
      ISUM=ISUM+NRCL
C
C6------IF STRT WILL BE SAVED THEN ALLOCATE SPACE.
   50 LCSTRT=ISUM
      IF(ISTRT.NE.0) ISUM=ISUM+NRCL
      ISP=ISUM-ISOLD
C
C7------PRINT AMOUNT OF SPACE USED.
      WRITE(IOUT,6) ISP
    6 FORMAT(1X,I10,' ELEMENTS IN X ARRAY ARE USED BY BAS')
      ISUM1=ISUM-1
      WRITE(IOUT,7) ISUM1,LENX
    7 FORMAT(1X,I10,' ELEMENTS OF X ARRAY USED OUT OF ',I10)
      IF(ISUM1.GT.LENX) WRITE(IOUT,8)
    8 FORMAT(1X,'   ***X ARRAY MUST BE DIMENSIONED LARGER***')
C
C
C8------RETURN
      RETURN
C
      END
      SUBROUTINE BAS5AD(DELT,TSMULT,TOTIM,PERTIM,HNEW,HOLD,KSTP,
     1                  NCOL,NROW,NLAY)
C
C-----VERSION 1412 22FEB1982 BAS5AD
C
C     ******************************************************************
C     ADVANCE TO NEXT TIME STEP
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DOUBLE PRECISION HNEW
C
      DIMENSION HNEW(NCOL,NROW,NLAY), HOLD(NCOL,NROW,NLAY)
C     ------------------------------------------------------------------
C
C1------IF NOT FIRST TIME STEP THEN CALCULATE TIME STEP LENGTH.
      IF(KSTP.NE.1) DELT=TSMULT*DELT
C
C2------ACCUMULATE ELAPSED TIME IN SIMULATION(TOTIM) AND IN THIS
C2------STRESS PERIOD(PERTIM).
      TOTIM=TOTIM+DELT
      PERTIM=PERTIM+DELT
C
C3------COPY HNEW TO HOLD.
      DO 10 K=1,NLAY
      DO 10 I=1,NROW
      DO 10 J=1,NCOL
   10 HOLD(J,I,K)=HNEW(J,I,K)
C
C4------RETURN
      RETURN
      END
      SUBROUTINE BAS5FM(HCOF,RHS,NODES)
C
C
C-----VERSION 1412 02JULY1993 BAS5FM
C     ******************************************************************
C     SET HCOF=RHS=0.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DIMENSION HCOF(NODES),RHS(NODES)
C     ------------------------------------------------------------------
C
C1------FOR EACH CELL INITIALIZE HCOF AND RHS ACCUMULATORS.
      ZERO=0.
      DO 100 I=1,NODES
      HCOF(I)=ZERO
      RHS(I)=ZERO
  100 CONTINUE
C
C2------RETURN
      RETURN
      END
      SUBROUTINE BAS5OC(NSTP,KSTP,ICNVG,IOFLG,NLAY,IBUDFL,ICBCFL,
     1      IHDDFL,INOC,IOUT,KPER,IPEROC,ITSOC,IBDOPT,IXSEC,IFREFM)
C
C-----VERSION 1340 20FEB1996 BAS5OC
C     ******************************************************************
C     OUTPUT CONTROLLER FOR HEAD, DRAWDOWN, AND BUDGET
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DIMENSION IOFLG(NLAY,4)
C     ------------------------------------------------------------------
C
C1------TEST UNIT NUMBER (INOC (INOC=IUNIT(12))) TO SEE IF
C1------OUTPUT CONTROL IS ACTIVE.  IF NOT, SET DEFAULTS AND RETURN.
      IF(INOC.LE.0) THEN
         IHDDFL=0
         IF(ICNVG.EQ.0 .OR. KSTP.EQ.NSTP)IHDDFL=1
         IBUDFL=0
         IF(ICNVG.EQ.0 .OR. KSTP.EQ.NSTP)IBUDFL=1
         ICBCFL=0
         GO TO 1000
      END IF
C
C2------OUTPUT CONTROL IS ACTIVE.  IF IPEROC >= 0, READ OUTPUT FLAGS
C2------USING ALPHABETIC INPUT STRUCTURE.
      IF(IPEROC.GE.0) THEN
         CALL SBAS5N(IOFLG,NLAY,IHDDFL,IBUDFL,ICBCFL,IPEROC,ITSOC,KPER,
     1            KSTP,INOC,IOUT,IBDOPT)
         GO TO 600
      END IF
C
C3------READ AND PRINT OUTPUT FLAGS AND CODE FOR DEFINING IOFLG USING
C3------THE ORIGINAL NUMERIC INPUT STRUCTURE.
      IF(IFREFM.EQ.0) THEN
         READ(INOC,'(4I10)') INCODE,IHDDFL,IBUDFL,ICBCFL
      ELSE
         READ(INOC,*) INCODE,IHDDFL,IBUDFL,ICBCFL
      END IF
      WRITE(IOUT,3) IHDDFL,IBUDFL,ICBCFL
    3 FORMAT(1X,/1X,'HEAD/DRAWDOWN PRINTOUT FLAG =',I2,
     1    5X,'TOTAL BUDGET PRINTOUT FLAG =',I2,
     2   /1X,'CELL-BY-CELL FLOW TERM FLAG =',I2)
      IF(ICBCFL.NE.0) ICBCFL=IBDOPT
C
C4------DECODE INCODE TO DETERMINE HOW TO SET FLAGS IN IOFLG.
      IF(INCODE) 100,200,300
C
C5------USE IOFLG FROM LAST TIME STEP.
  100 WRITE(IOUT,101)
  101 FORMAT(1X,'REUSING PREVIOUS VALUES OF IOFLG')
      GO TO 600
C
C6------READ IOFLG FOR LAYER 1 AND ASSIGN SAME TO ALL LAYERS
  200 IF(IFREFM.EQ.0) THEN
         READ(INOC,'(4I10)') (IOFLG(1,M),M=1,4)
      ELSE
         READ(INOC,*) (IOFLG(1,M),M=1,4)
      END IF
      DO 210 K=1,NLAY
      IOFLG(K,1)=IOFLG(1,1)
      IOFLG(K,2)=IOFLG(1,2)
      IOFLG(K,3)=IOFLG(1,3)
      IOFLG(K,4)=IOFLG(1,4)
  210 CONTINUE
      WRITE(IOUT,211) (IOFLG(1,M),M=1,4)
  211 FORMAT(1X,/1X,'OUTPUT FLAGS FOR ALL LAYERS ARE THE SAME:'/
     1   1X,'  HEAD    DRAWDOWN  HEAD  DRAWDOWN'/
     2   1X,'PRINTOUT  PRINTOUT  SAVE    SAVE'/
     3   1X,34('-')/1X,I5,I10,I8,I8)
      GO TO 600
C
C7------READ IOFLG IN ENTIRETY -- IF CROSS SECTION, READ ONLY ONE VALUE.
  300 IF(IXSEC.EQ.0) THEN
         DO 301 K=1,NLAY
         IF(IFREFM.EQ.0) THEN
            READ(INOC,'(4I10)') (IOFLG(K,M),M=1,4)
         ELSE
            READ(INOC,*) (IOFLG(K,M),M=1,4)
         END IF
  301    CONTINUE
         WRITE(IOUT,302) 'OUTPUT FLAGS FOR EACH LAYER:','LAYER'
  302    FORMAT(1X,/1X,A,/
     1   1X,'         HEAD    DRAWDOWN  HEAD  DRAWDOWN'/
     2   1X,A,'  PRINTOUT  PRINTOUT  SAVE    SAVE'/
     3   1X,41('-'))
         WRITE(IOUT,303) (K,(IOFLG(K,M),M=1,4),K=1,NLAY)
  303    FORMAT(1X,I4,I8,I10,I8,I8)
      ELSE
         IF(IFREFM.EQ.0) THEN
            READ(INOC,'(4I10)') (IOFLG(1,M),M=1,4)
         ELSE
            READ(INOC,*) (IOFLG(1,M),M=1,4)
         END IF
         WRITE(IOUT,302) 'OUTPUT FLAGS FOR CROSS SECTION:','     '
         WRITE(IOUT,304) (IOFLG(1,M),M=1,4)
  304    FORMAT(1X,I12,I10,I8,I8)
      END IF
C
C8------THE LAST STEP IN A STRESS PERIOD AND STEPS WHERE ITERATIVE
C8------PROCEDURE FAILED TO CONVERGE GET A VOLUMETRIC BUDGET.
  600 IF(ICNVG.EQ.0 .OR. KSTP.EQ.NSTP) IBUDFL=1
C
C9------RETURN
 1000 RETURN
C
      END
      SUBROUTINE BAS5OT(HNEW,STRT,ISTRT,BUFF,IOFLG,MSUM,IBOUND,VBNM,
     1  VBVL,KSTP,KPER,DELT,PERTIM,TOTIM,ITMUNI,NCOL,NROW,NLAY,ICNVG,
     2  IHDDFL,IBUDFL,IHEDFM,IHEDUN,IDDNFM,IDDNUN,IOUT,CHEDFM,CDDNFM,
     3  IXSEC,LBHDSV,LBDDSV,pdiffv,pdiffr)
C-----VERSION 1647 29OCT1992 BAS5OT
C     ******************************************************************
C     OUTPUT TIME, VOLUMETRIC BUDGET, HEAD, AND DRAWDOWN
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*16 VBNM(MSUM)
      DOUBLE PRECISION HNEW
      DIMENSION HNEW(NCOL,NROW,NLAY),STRT(NCOL,NROW,NLAY),
     1          VBVL(4,MSUM),IOFLG(NLAY,4),IBOUND(NCOL,NROW,NLAY),
     2          BUFF(NCOL,NROW,NLAY)
      CHARACTER*20 CHEDFM,CDDNFM
C     ------------------------------------------------------------------
C
C1------CLEAR PRINTOUT FLAG (IPFLG)
      IPFLG=0
C
C2------IF ITERATIVE PROCEDURE FAILED TO CONVERGE PRINT MESSAGE
      IF(ICNVG.EQ.0) WRITE(IOUT,1) KSTP,KPER
    1 FORMAT(1X,/11X,'****FAILED TO CONVERGE IN TIME STEP',I3,
     1      ' OF STRESS PERIOD',I3,'****')
C
C3------IF HEAD AND DRAWDOWN FLAG (IHDDFL) IS SET WRITE HEAD AND
C3------DRAWDOWN IN ACCORDANCE WITH FLAGS IN IOFLG.
      IF(IHDDFL.EQ.0) GO TO 100
C
      CALL SBAS5H(HNEW,BUFF,IOFLG,KSTP,KPER,NCOL,NROW,NLAY,IOUT,
     1    IHEDFM,IHEDUN,IPFLG,PERTIM,TOTIM,CHEDFM,IXSEC,LBHDSV,IBOUND)
      CALL SBAS5D(HNEW,BUFF,IOFLG,KSTP,KPER,NCOL,NROW,NLAY,IOUT,IDDNFM,
     1 IDDNUN,STRT,ISTRT,IBOUND,IPFLG,PERTIM,TOTIM,CDDNFM,IXSEC,
     2 LBDDSV)
C
C4------PRINT TOTAL BUDGET IF REQUESTED
  100 IF(IBUDFL.EQ.0) GO TO 120
      CALL SBAS5V(MSUM,VBNM,VBVL,KSTP,KPER,IOUT,pdiffv,pdiffr)
      IPFLG=1
C
C5------END PRINTOUT WITH TIME SUMMARY AND FORM FEED IF ANY PRINTOUT
C5------WILL BE PRODUCED.
  120 IF(IPFLG.EQ.0) RETURN
      CALL SBAS5T(KSTP,KPER,DELT,PERTIM,TOTIM,ITMUNI,IOUT)
      WRITE(IOUT,101)
  101 FORMAT('1')
C
C6------RETURN
      RETURN
      END
      SUBROUTINE BAS5RP(IBOUND,HNEW,STRT,HOLD,ISTRT,INBAS,HEADNG,NCOL,
     1    NROW,NLAY,VBVL,IOFLG,INOC,IHEDFM,IDDNFM,IHEDUN,IDDNUN,IOUT,
     2    IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,IXSEC,LBHDSV,LBDDSV,IFREFM)
C-----VERSION 1345 20FEB1996 BAS5RP
C     ******************************************************************
C     READ AND INITIALIZE BASIC MODEL ARRAYS
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*80 HEADNG(2)
      CHARACTER*24 ANAME(2)
      DOUBLE PRECISION HNEW,HNOFLO
      DIMENSION HNEW(NCOL,NROW,NLAY),IBOUND(NCOL,NROW,NLAY),
     1     STRT(NCOL,NROW,NLAY),HOLD(NCOL,NROW,NLAY),
     2     VBVL(4,40),IOFLG(NLAY,4)
      CHARACTER*20 CHEDFM,CDDNFM
C
      DATA ANAME(1) /'          BOUNDARY ARRAY'/
      DATA ANAME(2) /'            INITIAL HEAD'/
C     ------------------------------------------------------------------
C
C1------PRINT SIMULATION TITLE, CALCULATE # OF CELLS IN A LAYER.
      WRITE(IOUT,'(''1'',/1X,A)') HEADNG(1)
      WRITE(IOUT,'(1X,A)') HEADNG(2)
C
C2------READ BOUNDARY ARRAY(IBOUND) ONE LAYER AT A TIME.
      IF(IXSEC.EQ.0) THEN
         DO 100 K=1,NLAY
         KK=K
         CALL U2DINT(IBOUND(1,1,KK),ANAME(1),NROW,NCOL,KK,INBAS,IOUT)
  100    CONTINUE
      ELSE
         CALL U2DINT(IBOUND(1,1,1),ANAME(1),NLAY,NCOL,-1,INBAS,IOUT)
      END IF
C
C3------READ AND PRINT HEAD VALUE TO BE PRINTED FOR NO-FLOW CELLS.
      IF(IFREFM.EQ.0) THEN
         READ(INBAS,'(F10.0)') TMP
      ELSE
         READ(INBAS,*) TMP
      END IF
      HNOFLO=TMP
      WRITE(IOUT,3) TMP
    3 FORMAT(1X,/1X,'AQUIFER HEAD WILL BE SET TO ',1PG11.5,
     1       ' AT ALL NO-FLOW NODES (IBOUND=0).')
C
C4------READ INITIAL HEADS.
      IF(IXSEC.EQ.0) THEN
         DO 300 K=1,NLAY
         KK=K
         CALL U2DREL(HOLD(1,1,KK),ANAME(2),NROW,NCOL,KK,INBAS,IOUT)
  300    CONTINUE
      ELSE
         CALL U2DREL(HOLD(1,1,1),ANAME(2),NLAY,NCOL,-1,INBAS,IOUT)
      END IF
C
C5------COPY INITIAL HEADS FROM HOLD TO HNEW.
      DO 400 K=1,NLAY
      DO 400 I=1,NROW
      DO 400 J=1,NCOL
      HNEW(J,I,K)=HOLD(J,I,K)
      IF(IBOUND(J,I,K).EQ.0) HNEW(J,I,K)=HNOFLO
  400 CONTINUE
C
C6------IF STARTING HEADS ARE TO BE SAVED THEN COPY HOLD TO STRT.
      IF(ISTRT.EQ.0) GO TO 590
      DO 500 K=1,NLAY
      DO 500 I=1,NROW
      DO 500 J=1,NCOL
      STRT(J,I,K)=HOLD(J,I,K)
  500 CONTINUE
C
C7------INITIALIZE VOLUMETRIC BUDGET ACCUMULATORS TO ZERO.
  590 ZERO=0.
      DO 600 I=1,40
      DO 600 J=1,4
      VBVL(J,I)=ZERO
  600 CONTINUE
C
C8------SET UP OUTPUT CONTROL.
      CALL SBAS5I(NLAY,ISTRT,IOFLG,INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,
     1   IDDNUN,IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,IFREFM)
C
C9------RETURN
 1000 RETURN
      END
      SUBROUTINE BAS5ST(NSTP,DELT,TSMULT,PERTIM,KPER,INBAS,IOUT,IFREFM)
C
C
C-----VERSION 1418 20FEB1996 BAS5ST
C     ******************************************************************
C     SETUP TIME PARAMETERS FOR NEW TIME PERIOD
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
C     ------------------------------------------------------------------
C
C1------READ AND WRITE LENGTH OF STRESS PERIOD, NUMBER OF TIME STEPS AND
C1------TIME STEP MULTIPLIER.
      IF(IFREFM.EQ.0) THEN
         READ(INBAS,'(F10.0,I10,F10.0)') PERLEN,NSTP,TSMULT
      ELSE
         READ(INBAS,*) PERLEN,NSTP,TSMULT
      END IF
      WRITE (IOUT,1) KPER,PERLEN,NSTP,TSMULT
    1 FORMAT('1',/28X,'STRESS PERIOD NO.',I4,', LENGTH =',G15.7,/
     1            28X,46('-'),//
     2            30X,'NUMBER OF TIME STEPS =',I6,//
     3            31X,'MULTIPLIER FOR DELT =',F10.3)
C
C1A-----STOP IF NSTP LE 0, PERLEN LE 0., OR TSMULT LE 0.
      IF(NSTP.LE.0) THEN
         WRITE(IOUT,2)
    2    FORMAT(1X,/1X,'THERE MUST BE AT LEAST ONE TIME STEP')
         STOP
      END IF
      ZERO=0.
      IF(PERLEN.LE.ZERO) THEN
         WRITE(IOUT,3)
    3    FORMAT(1X,/1X,'PERLEN MUST BE GREATER THAN 0.0')
         STOP
      END IF
      IF(TSMULT.LE.ZERO) THEN
         WRITE(IOUT,4)
    4    FORMAT(1X,/1X,'TSMULT MUST BE GREATER THAN 0.0')
         STOP
      END IF
C
C2------CALCULATE THE LENGTH OF THE FIRST TIME STEP.
C
C2A-----ASSUME TIME STEP MULTIPLIER IS EQUAL TO ONE.
      DELT=PERLEN/FLOAT(NSTP)
C
C2B-----IF TIME STEP MULTIPLIER IS NOT ONE THEN CALCULATE FIRST
C2B-----TERM OF GEOMETRIC PROGRESSION.
      ONE=1.
      IF(TSMULT.NE.ONE) DELT=PERLEN*(ONE-TSMULT)/(ONE-TSMULT**NSTP)
C
C3------PRINT THE LENGTH OF THE FIRST TIME STEP.
      WRITE (IOUT,9) DELT
    9 FORMAT(1X,/28X,'INITIAL TIME STEP SIZE =',G15.7)
C
C4------INITIALIZE PERTIM (ELAPSED TIME WITHIN STRESS PERIOD).
      PERTIM=0.
C
C5------RETURN
      RETURN
      END
      SUBROUTINE SBAS5D(HNEW,BUFF,IOFLG,KSTP,KPER,NCOL,NROW,
     1    NLAY,IOUT,IDDNFM,IDDNUN,STRT,ISTRT,IBOUND,IPFLG,
     2    PERTIM,TOTIM,CDDNFM,IXSEC,LBDDSV)
C-----VERSION 1648 25JUNE1993 SBAS5D
C     ******************************************************************
C     CALCULATE, PRINT, AND SAVE DRAWDOWNS
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*16 TEXT
      DOUBLE PRECISION HNEW,SSTRT
      DIMENSION HNEW(NCOL,NROW,NLAY),IOFLG(NLAY,4),
     1     BUFF(NCOL,NROW,NLAY),STRT(NCOL,NROW,NLAY),
     1     IBOUND(NCOL,NROW,NLAY)
      CHARACTER*20 CDDNFM
C
      DATA TEXT /'        DRAWDOWN'/
C     ------------------------------------------------------------------
C
C1------FOR EACH LAYER CALCULATE DRAWDOWN IF PRINT OR SAVE IS REQUESTED.
      DO 59 K=1,NLAY
C
C2------IS DRAWDOWN NEEDED FOR THIS LAYER?
      KL=K
      IF(IXSEC.NE.0) KL=1
      IF(IOFLG(KL,2).EQ.0 .AND. IOFLG(KL,4).EQ.0) GO TO 59
C
C3------DRAWDOWN IS NEEDED. WERE INITIAL HEADS KEPT?
      IF(ISTRT.EQ.0) THEN
         WRITE(IOUT,52)
   52    FORMAT(1X,/1X,'CANNOT CALCULATE DRAWDOWN BECAUSE INITIAL HEAD',
     1   ' WAS NOT KEPT AFTER THE'/1X,
     2   'SIMULATION STARTED.  SEE "ISTRT" PARAMETER IN BAS INPUT.')
         STOP
      END IF
C
C4------CALCULATE DRAWDOWN FOR THE LAYER.
      DO 58 I=1,NROW
      DO 58 J=1,NCOL
      BUFF(J,I,K)=HNEW(J,I,K)
      SSTRT=STRT(J,I,K)
      IF(IBOUND(J,I,K).NE.0) BUFF(J,I,K)=SSTRT-HNEW(J,I,K)
   58 CONTINUE
   59 CONTINUE
C
C5------FOR EACH LAYER: DETERMINE IF DRAWDOWN SHOULD BE PRINTED.
C5------IF SO THEN CALL ULAPRS OR ULAPRW TO PRINT DRAWDOWN.
      IF(IXSEC.EQ.0) THEN
        DO 69 K=1,NLAY
        KK=K
        IF(IOFLG(K,2).EQ.0) GO TO 69
        IF(IDDNFM.LT.0) CALL ULAPRS(BUFF(1,1,K),TEXT,KSTP,KPER,
     1               NCOL,NROW,KK,-IDDNFM,IOUT)
        IF(IDDNFM.GE.0) CALL ULAPRW(BUFF(1,1,K),TEXT,KSTP,KPER,
     1               NCOL,NROW,KK,IDDNFM,IOUT)
        IPFLG=1
   69   CONTINUE
C
C5A-----PRINT DRAWDOWN FOR CROSS SECTION.
      ELSE
        IF(IOFLG(1,2).NE.0) THEN
          IF(IDDNFM.LT.0) CALL ULAPRS(BUFF,TEXT,KSTP,KPER,
     1                 NCOL,NLAY,-1,-IDDNFM,IOUT)
          IF(IDDNFM.GE.0) CALL ULAPRW(BUFF,TEXT,KSTP,KPER,
     1                 NCOL,NLAY,-1,IDDNFM,IOUT)
          IPFLG=1
        END IF
      END IF
C
C6------FOR EACH LAYER: DETERMINE IF DRAWDOWN SHOULD BE SAVED.
C6------IF SO THEN CALL A ULASAV OR ULASV2 TO RECORD DRAWDOWN.
      IFIRST=1
      IF(IDDNUN.LE.0) GO TO 80
      IF(IXSEC.EQ.0) THEN
        DO 79 K=1,NLAY
        KK=K
        IF(IOFLG(K,4).EQ.0) GO TO 79
        IF(IFIRST.EQ.1) WRITE(IOUT,74) IDDNUN,KSTP,KPER
   74   FORMAT(1X,/1X,'DRAWDOWN WILL BE SAVED ON UNIT',I4,
     1      ' AT END OF TIME STEP',I3,', STRESS PERIOD',I3)
        IFIRST=0
        IF(CDDNFM.EQ.' ') THEN
           CALL ULASAV(BUFF(1,1,K),TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NROW,KK,IDDNUN)
        ELSE
           CALL ULASV2(BUFF(1,1,K),TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NROW,KK,IDDNUN,CDDNFM,LBDDSV,IBOUND(1,1,K))
        END IF
   79   CONTINUE
C
C6A-----SAVE DRAWDOWN FOR CROSS SECTION.
      ELSE
        IF(IOFLG(1,4).NE.0) THEN
          WRITE(IOUT,74) IDDNUN,KSTP,KPER
          IF(CDDNFM.EQ.' ') THEN
             CALL ULASAV(BUFF,TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NLAY,-1,IDDNUN)
          ELSE
             CALL ULASV2(BUFF,TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                  NLAY,-1,IDDNUN,CDDNFM,LBDDSV,IBOUND)
          END IF
        END IF
      END IF
C
C7------RETURN.
   80 RETURN
      END
      SUBROUTINE SBAS5H(HNEW,BUFF,IOFLG,KSTP,KPER,NCOL,NROW,NLAY,IOUT,
     1   IHEDFM,IHEDUN,IPFLG,PERTIM,TOTIM,CHEDFM,IXSEC,LBHDSV,IBOUND)
C
C-----VERSION 1647 18OCT1993 SBAS5H
C     ******************************************************************
C     PRINT AND RECORD HEADS
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*16 TEXT
      DOUBLE PRECISION HNEW
      DIMENSION HNEW(NCOL,NROW,NLAY),IOFLG(NLAY,4),BUFF(NCOL,NROW,NLAY),
     1   IBOUND(NCOL,NROW,NLAY)
      CHARACTER*20 CHEDFM
C
      DATA TEXT /'            HEAD'/
C     ------------------------------------------------------------------
C
C1------FOR EACH LAYER MOVE HNEW TO BUFF IF PRINT OR SAVE IS REQUESTED.
      DO 59 K=1,NLAY
C
C2------IS HEAD NEEDED FOR THIS LAYER?
      KL=K
      IF(IXSEC.NE.0) KL=1
      IF(IOFLG(KL,1).EQ.0 .AND. IOFLG(KL,3).EQ.0) GO TO 59
C
C3------MOVE HNEW TO BUFF FOR THE LAYER.
      DO 58 I=1,NROW
      DO 58 J=1,NCOL
      BUFF(J,I,K)=HNEW(J,I,K)
   58 CONTINUE
   59 CONTINUE
C
C4------FOR EACH LAYER: DETERMINE IF HEAD SHOULD BE PRINTED.
C4------IF SO THEN CALL ULAPRS OR ULAPRW TO PRINT HEAD.
      IF(IXSEC.EQ.0) THEN
        DO 69 K=1,NLAY
        KK=K
        IF(IOFLG(K,1).EQ.0) GO TO 69
        IF(IHEDFM.LT.0) CALL ULAPRS(BUFF(1,1,K),TEXT,KSTP,KPER,
     1               NCOL,NROW,KK,-IHEDFM,IOUT)
        IF(IHEDFM.GE.0) CALL ULAPRW(BUFF(1,1,K),TEXT,KSTP,KPER,
     1               NCOL,NROW,KK,IHEDFM,IOUT)
        IPFLG=1
   69   CONTINUE
C
C4A-----PRINT HEAD FOR CROSS SECTION.
      ELSE
        IF(IOFLG(1,1).NE.0) THEN
          IF(IHEDFM.LT.0) CALL ULAPRS(BUFF,TEXT,KSTP,KPER,
     1                 NCOL,NLAY,-1,-IHEDFM,IOUT)
          IF(IHEDFM.GE.0) CALL ULAPRW(BUFF,TEXT,KSTP,KPER,
     1                 NCOL,NLAY,-1,IHEDFM,IOUT)
          IPFLG=1
        END IF
      END IF
C
C5------FOR EACH LAYER: DETERMINE IF HEAD SHOULD BE SAVED ON DISK.
C5------IF SO THEN CALL ULASAV OR ULASV2 TO SAVE HEAD.
      IFIRST=1
      IF(IHEDUN.LE.0) GO TO 80
      IF(IXSEC.EQ.0) THEN
        DO 79 K=1,NLAY
        KK=K
        IF(IOFLG(K,3).EQ.0) GO TO 79
        IF(IFIRST.EQ.1) WRITE(IOUT,74) IHEDUN,KSTP,KPER
   74   FORMAT(1X,/1X,'HEAD WILL BE SAVED ON UNIT',I4,
     1      ' AT END OF TIME STEP',I3,', STRESS PERIOD',I3)
        IFIRST=0
        IF(CHEDFM.EQ.' ') THEN
           CALL ULASAV(BUFF(1,1,K),TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NROW,KK,IHEDUN)
        ELSE
           CALL ULASV2(BUFF(1,1,K),TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NROW,KK,IHEDUN,CHEDFM,LBHDSV,IBOUND(1,1,K))
        END IF
   79   CONTINUE
C
C5A-----SAVE HEAD FOR CROSS SECTION.
      ELSE
        IF(IOFLG(1,3).NE.0) THEN
          WRITE(IOUT,74) IHEDUN,KSTP,KPER
          IF(CHEDFM.EQ.' ') THEN
             CALL ULASAV(BUFF,TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                NLAY,-1,IHEDUN)
          ELSE
             CALL ULASV2(BUFF,TEXT,KSTP,KPER,PERTIM,TOTIM,NCOL,
     1                  NLAY,-1,IHEDUN,CHEDFM,LBHDSV,IBOUND)
          END IF
        END IF
      END IF
C
C6------RETURN.
   80 RETURN
      END
      SUBROUTINE SBAS5I(NLAY,ISTRT,IOFLG,INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,
     1   IDDNUN,IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,IFREFM)
C
C-----VERSION 1520 20FEB1996 SBAS5I
C     ******************************************************************
C     SET UP OUTPUT CONTROL.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DIMENSION IOFLG(NLAY,4)
      CHARACTER*20 CHEDFM,CDDNFM
      CHARACTER*80 LINE
C     ------------------------------------------------------------------
C
C1------ASSIGN DEFAULT VALUES.
      CHEDFM=' '
      CDDNFM=' '
      IHEDFM=0
      IDDNFM=0
      IHEDUN=0
      IDDNUN=0
      IBDOPT=1
      LBHDSV=0
      LBDDSV=0
C
C2------TEST OUTPUT CONTROL INPUT UNIT TO SEE IF OUTPUT CONTROL IS
C2------ACTIVE.
      IF(INOC.LE.0) THEN
C
C2A-----OUTPUT CONTROL IS INACTIVE. PRINT A MESSAGE LISTING DEFAULTS.
         WRITE(IOUT, 41)
   41    FORMAT(1X,/1X,'DEFAULT OUTPUT CONTROL',/1X,
     1  'THE FOLLOWING OUTPUT COMES AT THE END OF EACH STRESS PERIOD:')
         WRITE(IOUT, 42)
   42    FORMAT(1X,'TOTAL VOLUMETRIC BUDGET')
         WRITE(IOUT, 43)
   43    FORMAT(1X,10X,'HEAD')
         IF(ISTRT.NE.0) WRITE(IOUT, 44)
   44    FORMAT(1X,10X,'DRAWDOWN')
C
C2B-----SET DEFAULT FLAGS IN IOFLG SO THAT HEAD (AND DRAWDOWN) IS
C2B-----PRINTED FOR EVERY LAYER.
         ID=0
         IF(ISTRT.NE.0) ID=1
         DO 80 K=1,NLAY
         IOFLG(K,1)=1
         IOFLG(K,2)=ID
         IOFLG(K,3)=0
         IOFLG(K,4)=0
   80    CONTINUE
         GO TO 1000
      END IF
C
C3------OUTPUT CONTROL IS ACTIVE.  READ FIRST RECORD AND DECODE FIRST
C3------WORD.  MUST USE URWORDd IN CASE FIRST WORD IS ALPHABETIC.
      READ(INOC,'(A)') LINE
      LLOC=1
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
C
C4------TEST FOR NUMERIC OUTPUT CONTROL.  FIRST WORD WILL NOT BE
C4------"PERIOD", "HEAD", "DRAWDOWN", OR "COMPACT".
      IF(LINE(ISTART:ISTOP).NE.'PERIOD' .AND. LINE(ISTART:ISTOP).NE.
     1     'HEAD' .AND. LINE(ISTART:ISTOP).NE.'DRAWDOWN' .AND.
     2     LINE(ISTART:ISTOP).NE.'COMPACT') THEN
C4A-----NUMERIC OUTPUT CONTROL.  DECODE THE INITIAL RECORD ACCORDINGLY.
         WRITE(IOUT,102)
  102    FORMAT(1X,/1X,'OUTPUT CONTROL IS SPECIFIED EVERY TIME STEP')
         IF(IFREFM.EQ.0) THEN
            READ(LINE,'(4I10)') IHEDFM,IDDNFM,IHEDUN,IDDNUN
         ELSE
            LLOC=1
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDFM,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNFM,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDUN,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNUN,R,IOUT,INOC)
         END IF
         WRITE(IOUT,103) IHEDFM,IDDNFM
  103    FORMAT(1X,'HEAD PRINT FORMAT CODE IS',I4,
     1     '    DRAWDOWN PRINT FORMAT CODE IS',I4)
         WRITE(IOUT,104) IHEDUN,IDDNUN
  104    FORMAT(1X,'HEADS WILL BE SAVED ON UNIT',I4,
     1     '    DRAWDOWNS WILL BE SAVED ON UNIT',I4)
         IPEROC=-1
         ITSOC=-1
      ELSE
C4B-----ALPHABETIC OUTPUT CONTROL.  CALL MODULE TO READ INITIAL RECORDS.
         CALL SBAS5J(INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,IDDNUN,
     1         IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,
     2         LINE,LLOC,ISTART,ISTOP)
      END IF
C
C5------RETURN.
 1000 RETURN
      END
      SUBROUTINE SBAS5J(INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,IDDNUN,
     1         IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,
     2         LINE,LLOC,ISTART,ISTOP)
C
C-----VERSION 1433 3JAN1995 SBAS5J
C     ******************************************************************
C     READ INITIAL ALPHABETIC OUTPUT CONTROL RECORDS.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*20 CHEDFM,CDDNFM
      CHARACTER*80 LINE
C     ------------------------------------------------------------------
C
C1------ALPHABETIC OUTPUT CONTROL.  WRITE MESSAGE AND SET INITIAL VALUES
C1------FOR IPEROC AND ITSOC.
      WRITE(IOUT,91)
   91 FORMAT(1X,/1X,'OUTPUT CONTROL IS SPECIFIED ONLY AT TIME STEPS',
     1    ' FOR WHICH OUTPUT IS DESIRED')
      IPEROC=9999
      ITSOC=9999
C
C2------LOOK FOR ALPHABETIC WORDS:
C2A-----LOOK FOR "PERIOD", WHICH INDICATES THE END OF INITIAL OUTPUT
C2A-----CONTROL DATA.  IF FOUND, DECODE THE PERIOD NUMBER AND TIME
C2A-----STEP NUMBER FOR LATER USE.
  100 IF(LINE(ISTART:ISTOP).EQ.'PERIOD') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IPEROC,R,IOUT,INOC)
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).NE.'STEP') GO TO 2000
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,ITSOC,R,IOUT,INOC)
         WRITE(IOUT,101) IHEDFM,IDDNFM
  101    FORMAT(1X,'HEAD PRINT FORMAT CODE IS',I4,
     1        '    DRAWDOWN PRINT FORMAT CODE IS',I4)
         WRITE(IOUT,102) IHEDUN,IDDNUN
  102    FORMAT(1X,'HEADS WILL BE SAVED ON UNIT',I4,
     1        '    DRAWDOWNS WILL BE SAVED ON UNIT',I4)
         GO TO 1000
C
C2B-----LOOK FOR "HEAD PRINT FORMAT" AND "HEAD SAVE FORMAT".  IF
C2B-----FOUND, SET APPROPRIATE FLAGS.
      ELSE IF(LINE(ISTART:ISTOP).EQ.'HEAD') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).EQ.'PRINT') THEN
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
            IF(LINE(ISTART:ISTOP).NE.'FORMAT') GO TO 2000
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDFM,R,IOUT,INOC)
         ELSE IF(LINE(ISTART:ISTOP).EQ.'SAVE') THEN
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
            IF(LINE(ISTART:ISTOP).EQ.'UNIT') THEN
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDUN,R,IOUT,
     1            INOC)
            ELSE IF(LINE(ISTART:ISTOP).EQ.'FORMAT') THEN
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,0,N,R,IOUT,INOC)
               CHEDFM=LINE(ISTART:ISTOP)
               WRITE(IOUT,103) CHEDFM
  103          FORMAT(1X,'HEADS WILL BE SAVED WITH FORMAT: ',A)
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
               IF(LINE(ISTART:ISTOP).EQ.'LABEL') THEN
                  LBHDSV=1
                  WRITE(IOUT,104)
  104             FORMAT(1X,'SAVED HEADS WILL BE LABELED')
               END IF
            ELSE
               GO TO 2000
            END IF
         ELSE
            GO TO 2000
         END IF
C
C2C-----LOOK FOR "DRAWDOWN PRINT FORMAT" AND "DRAWDOWN SAVE FORMAT".
C2C-----IF FOUND, SET APPROPRIATE FLAGS
      ELSE IF(LINE(ISTART:ISTOP).EQ.'DRAWDOWN') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).EQ.'PRINT') THEN
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
            IF(LINE(ISTART:ISTOP).NE.'FORMAT') GO TO 2000
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNFM,R,IOUT,INOC)
         ELSE IF(LINE(ISTART:ISTOP).EQ.'SAVE') THEN
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
            IF(LINE(ISTART:ISTOP).EQ.'UNIT') THEN
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNUN,R,IOUT,
     1                   INOC)
            ELSE IF(LINE(ISTART:ISTOP).EQ.'FORMAT') THEN
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,0,N,R,IOUT,INOC)
               CDDNFM=LINE(ISTART:ISTOP)
               WRITE(IOUT,105) CDDNFM
  105          FORMAT(1X,'DRAWDOWN WILL BE SAVED WITH FORMAT: ',A)
               CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
               IF(LINE(ISTART:ISTOP).EQ.'LABEL') THEN
                  LBDDSV=1
                  WRITE(IOUT,106)
  106             FORMAT(1X,'SAVED DRAWDOWN WILL BE LABELED')
               END IF
            ELSE
               GO TO 2000
            END IF
         ELSE
            GO TO 2000
         END IF
C
C2D-----LOOK FOR "COMPACT BUDGET FILES" -- "COMPACT" IS SUFFICIENT.
C2D-----IF FOUND, SET APPROPRIATE FLAG.
      ELSE IF(LINE(ISTART:ISTOP).EQ.'COMPACT') THEN
         IBDOPT=2
         WRITE(IOUT,107)
  107    FORMAT(1X,'COMPACT CELL-BY-CELL BUDGET FILES WILL BE WRITTEN')
C
C2E-----ERROR IF UNRECOGNIZED WORD.
      ELSE
         GO TO 2000
      END IF
C
C3------FINISHED READING A RECORD.  READ NEXT RECORD, IGNORING BLANK
C3------LINES.  GO BACK AND DECODE IT.
  110 READ(INOC,'(A)',END=1000) LINE
      IF(LINE.EQ.' ') GO TO 110
      LLOC=1
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
      GO TO 100
C
C4------RETURN.
 1000 RETURN
C
C5------ERROR DECODING INPUT DATA.
 2000 WRITE(IOUT,2001) LINE
 2001 FORMAT(1X,/1X,'ERROR READING OUTPUT CONTROL INPUT DATA:'/1X,A80)
      STOP
      END
      SUBROUTINE SBAS5T(KSTP,KPER,DELT,PERTIM,TOTIM,ITMUNI,IOUT)
C
C
C-----VERSION 0959 22JUNE1992 SBAS5T
C     ******************************************************************
C     PRINT SIMULATION TIME
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
C     ------------------------------------------------------------------
      WRITE(IOUT,199) KSTP,KPER
  199 FORMAT(1X,///10X,'TIME SUMMARY AT END OF TIME STEP',I3,
     1     ' IN STRESS PERIOD',I3)
C
C1------USE TIME UNIT INDICATOR TO GET FACTOR TO CONVERT TO SECONDS.
      ZERO=0.
      CNV=ZERO
      IF(ITMUNI.EQ.1) CNV=1.
      IF(ITMUNI.EQ.2) CNV=60.
      IF(ITMUNI.EQ.3) CNV=3600.
      IF(ITMUNI.EQ.4) CNV=86400.
      IF(ITMUNI.EQ.5) CNV=31557600.
C
C2------IF FACTOR=0 THEN TIME UNITS ARE NON-STANDARD.
      IF(CNV.NE.ZERO) GO TO 100
C
C2A-----PRINT TIMES IN NON-STANDARD TIME UNITS.
      WRITE(IOUT,301) DELT,PERTIM,TOTIM
  301 FORMAT(21X,'     TIME STEP LENGTH =',G15.6/
     1       21X,'   STRESS PERIOD TIME =',G15.6/
     2       21X,'TOTAL SIMULATION TIME =',G15.6)
C
C2B-----RETURN
      RETURN
C
C3------CALCULATE LENGTH OF TIME STEP & ELAPSED TIMES IN SECONDS.
  100 DELSEC=CNV*DELT
      TOTSEC=CNV*TOTIM
      PERSEC=CNV*PERTIM
C
C4------CALCULATE TIMES IN MINUTES,HOURS,DAYS AND YEARS.
      SIXTY=60.
      HRDAY=24.
      DAYYR=365.25
      DELMN=DELSEC/SIXTY
      DELHR=DELMN/SIXTY
      DELDY=DELHR/HRDAY
      DELYR=DELDY/DAYYR
      TOTMN=TOTSEC/SIXTY
      TOTHR=TOTMN/SIXTY
      TOTDY=TOTHR/HRDAY
      TOTYR=TOTDY/DAYYR
      PERMN=PERSEC/SIXTY
      PERHR=PERMN/SIXTY
      PERDY=PERHR/HRDAY
      PERYR=PERDY/DAYYR
C
C5------PRINT TIME STEP LENGTH AND ELAPSED TIMES IN ALL TIME UNITS.
      WRITE(IOUT,200)
  200 FORMAT(19X,' SECONDS     MINUTES      HOURS',7X,
     1    'DAYS        YEARS'/20X,59('-'))
      WRITE (IOUT,201) DELSEC,DELMN,DELHR,DELDY,DELYR
  201 FORMAT(1X,'  TIME STEP LENGTH',1P,5G12.5)
      WRITE(IOUT,202) PERSEC,PERMN,PERHR,PERDY,PERYR
  202 FORMAT(1X,'STRESS PERIOD TIME',1P,5G12.5)
      WRITE(IOUT,203) TOTSEC,TOTMN,TOTHR,TOTDY,TOTYR
  203 FORMAT(1X,'        TOTAL TIME',1P,5G12.5)
C
C6------RETURN
      RETURN
      END
      SUBROUTINE SBAS5V(MSUM,VBNM,VBVL,KSTP,KPER,IOUT,pdiffv,pdiffr)
C
C
C-----VERSION 1448 28APRIL1994 SBAS5V
C     ******************************************************************
C     PRINT VOLUMETRIC BUDGET
C     ******************************************************************
C
C     SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      CHARACTER*16 VBNM(MSUM)
      DIMENSION VBVL(4,MSUM)
      CHARACTER*17 VAL1,VAL2
C     ------------------------------------------------------------------
C
C1------DETERMINE NUMBER OF INDIVIDUAL BUDGET ENTRIES.
      MSUM1=MSUM-1
      IF(MSUM1.LE.0) RETURN
C
C2------CLEAR RATE AND VOLUME ACCUMULATORS.
      ZERO=0.
      TWO=2.
      HUND=100.
      BIGVL1=9.99999E11
      BIGVL2=9.99999E10
      SMALL=0.1
      TOTRIN=ZERO
      TOTROT=ZERO
      TOTVIN=ZERO
      TOTVOT=ZERO
C
C3------ADD RATES AND VOLUMES (IN AND OUT) TO ACCUMULATORS.
      DO 100 L=1,MSUM1
      TOTRIN=TOTRIN+VBVL(3,L)
      TOTROT=TOTROT+VBVL(4,L)
      TOTVIN=TOTVIN+VBVL(1,L)
      TOTVOT=TOTVOT+VBVL(2,L)
  100 CONTINUE
C
C4------PRINT TIME STEP NUMBER AND STRESS PERIOD NUMBER.
      WRITE(IOUT,260) KSTP,KPER
      WRITE(IOUT,265)
C
C5------PRINT INDIVIDUAL INFLOW RATES AND VOLUMES AND THEIR TOTALS.
      DO 200 L=1,MSUM1
      IF(VBVL(1,L).NE.ZERO .AND.
     1       (VBVL(1,L).GE.BIGVL1 .OR. VBVL(1,L).LT.SMALL)) THEN
         WRITE(VAL1,'(1PE17.4)') VBVL(1,L)
      ELSE
         WRITE(VAL1,'(F17.4)') VBVL(1,L)
      END IF
      IF(VBVL(3,L).NE.ZERO .AND.
     1       (VBVL(3,L).GE.BIGVL1 .OR. VBVL(3,L).LT.SMALL)) THEN
         WRITE(VAL2,'(1PE17.4)') VBVL(3,L)
      ELSE
         WRITE(VAL2,'(F17.4)') VBVL(3,L)
      END IF
      WRITE(IOUT,275) VBNM(L),VAL1,VBNM(L),VAL2
  200 CONTINUE
      IF(TOTVIN.NE.ZERO .AND.
     1      (TOTVIN.GE.BIGVL1 .OR. TOTVIN.LT.SMALL)) THEN
         WRITE(VAL1,'(1PE17.4)') TOTVIN
      ELSE
         WRITE(VAL1,'(F17.4)') TOTVIN
      END IF
      IF(TOTRIN.NE.ZERO .AND.
     1      (TOTRIN.GE.BIGVL1 .OR. TOTRIN.LT.SMALL)) THEN
         WRITE(VAL2,'(1PE17.4)') TOTRIN
      ELSE
         WRITE(VAL2,'(F17.4)') TOTRIN
      END IF
      WRITE(IOUT,286) VAL1,VAL2
C
C6------PRINT INDIVIDUAL OUTFLOW RATES AND VOLUMES AND THEIR TOTALS.
      WRITE(IOUT,287)
      DO 250 L=1,MSUM1
      IF(VBVL(2,L).NE.ZERO .AND.
     1       (VBVL(2,L).GE.BIGVL1 .OR. VBVL(2,L).LT.SMALL)) THEN
         WRITE(VAL1,'(1PE17.4)') VBVL(2,L)
      ELSE
         WRITE(VAL1,'(F17.4)') VBVL(2,L)
      END IF
      IF(VBVL(4,L).NE.ZERO .AND.
     1       (VBVL(4,L).GE.BIGVL1 .OR. VBVL(4,L).LT.SMALL)) THEN
         WRITE(VAL2,'(1PE17.4)') VBVL(4,L)
      ELSE
         WRITE(VAL2,'(F17.4)') VBVL(4,L)
      END IF
      WRITE(IOUT,275) VBNM(L),VAL1,VBNM(L),VAL2
  250 CONTINUE
      IF(TOTVOT.NE.ZERO .AND.
     1      (TOTVOT.GE.BIGVL1 .OR. TOTVOT.LT.SMALL)) THEN
         WRITE(VAL1,'(1PE17.4)') TOTVOT
      ELSE
         WRITE(VAL1,'(F17.4)') TOTVOT
      END IF
      IF(TOTROT.NE.ZERO .AND.
     1      (TOTROT.GE.BIGVL1 .OR. TOTROT.LT.SMALL)) THEN
         WRITE(VAL2,'(1PE17.4)') TOTROT
      ELSE
         WRITE(VAL2,'(F17.4)') TOTROT
      END IF
      WRITE(IOUT,298) VAL1,VAL2
C
C7------CALCULATE THE DIFFERENCE BETWEEN INFLOW AND OUTFLOW.
C
C7A-----CALCULATE DIFFERENCE BETWEEN RATE IN AND RATE OUT.
      DIFFR=TOTRIN-TOTROT
      ADIFFR=ABS(DIFFR)
C
C7B-----CALCULATE PERCENT DIFFERENCE BETWEEN RATE IN AND RATE OUT.
      PDIFFR=ZERO
      AVGRAT=(TOTRIN+TOTROT)/TWO
      IF(AVGRAT.NE.ZERO) PDIFFR=HUND*DIFFR/AVGRAT
C
C7C-----CALCULATE DIFFERENCE BETWEEN VOLUME IN AND VOLUME OUT.
      DIFFV=TOTVIN-TOTVOT
      ADIFFV=ABS(DIFFV)
C
C7D-----GET PERCENT DIFFERENCE BETWEEN VOLUME IN AND VOLUME OUT.
      PDIFFV=ZERO
      AVGVOL=(TOTVIN+TOTVOT)/TWO
      IF(AVGVOL.NE.ZERO) PDIFFV=HUND*DIFFV/AVGVOL
C
C8------PRINT DIFFERENCES AND PERCENT DIFFERENCES BETWEEN INPUT
C8------AND OUTPUT RATES AND VOLUMES.
      IF(ADIFFV.NE.ZERO .AND.
     1      (ADIFFV.GE.BIGVL2 .OR. ADIFFV.LT.SMALL)) THEN
         WRITE(VAL1,'(1PE17.4)') DIFFV
      ELSE
         WRITE(VAL1,'(F17.4)') DIFFV
      END IF
      IF(ADIFFR.NE.ZERO .AND.
     1      (ADIFFR.GE.BIGVL2 .OR. ADIFFR.LT.SMALL)) THEN
         WRITE(VAL2,'(1PE17.4)') DIFFR
      ELSE
         WRITE(VAL2,'(F17.4)') DIFFR
      END IF
      WRITE(IOUT,299) VAL1,VAL2
      WRITE(IOUT,300) PDIFFV,PDIFFR
C
C9------RETURN.
      RETURN
C
C    ---FORMATS
C
  260 FORMAT('1',/2X,'VOLUMETRIC BUDGET FOR ENTIRE MODEL AT END OF'
     1,' TIME STEP',I3,' IN STRESS PERIOD',I3/2X,77('-'))
  265 FORMAT(1X,/5X,'CUMULATIVE VOLUMES',6X,'L**3',7X
     1,'RATES FOR THIS TIME STEP',6X,'L**3/T'/5X,18('-'),17X,24('-')
     2//11X,'IN:',38X,'IN:'/11X,'---',38X,'---')
  275 FORMAT(1X,3X,A16,' =',A17,6X,A16,' =',A17)
  286 FORMAT(1X,/12X,'TOTAL IN =',A,14X,'TOTAL IN =',A)
  287 FORMAT(1X,/10X,'OUT:',37X,'OUT:'/10X,4('-'),37X,4('-'))
  298 FORMAT(1X,/11X,'TOTAL OUT =',A,13X,'TOTAL OUT =',A)
  299 FORMAT(1X,/12X,'IN - OUT =',A,14X,'IN - OUT =',A)
  300 FORMAT(1X,/1X,'PERCENT DISCREPANCY =',F15.2
     1,5X,'PERCENT DISCREPANCY =',F15.2,///)
C
      END
      SUBROUTINE SBAS5N(IOFLG,NLAY,IHDDFL,IBUDFL,ICBCFL,IPEROC,ITSOC,
     1            KPER,KSTP,INOC,IOUT,IBDOPT)
C
C-----VERSION 0932 14FEB1994 SBAS5N
C     ******************************************************************
C     SET OUTPUT FLAGS USING ALPHABETIC OUTPUT CONTROL INPUT STRUCTURE
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DIMENSION IOFLG(NLAY,4)
      CHARACTER*80 LINE
C     ------------------------------------------------------------------
C
C1------ERROR IF OUTPUT CONTROL TIME STEP PRECEDES CURRENT SIMULATION
C1------TIME STEP.
      IF((IPEROC.LT.KPER).OR.(IPEROC.EQ.KPER .AND. ITSOC.LT.KSTP)) THEN
         WRITE(IOUT,5) IPEROC,ITSOC,KPER,KSTP
    5    FORMAT(1X,/1X,'OUTPUT CONTROL WAS SPECIFIED FOR A NONEXISTENT',
     1   ' TIME STEP',/
     2   1X,'OR OUTPUT CONTROL DATA ARE NOT ENTERED IN ASCENDING ORDER',
     3   /1X,'OUTPUT CONTROL STRESS PERIOD',I3,'   TIME STEP',I3,/
     4   1X,'MODEL STRESS PERIOD',I3,'   TIME STEP',I3,/
     5   1X,'APPLYING THE SPECIFIED OUTPUT CONTROL TO THE CURRENT TIME',
     6   ' STEP')
         IPEROC=KPER
         ITSOC=KSTP
      END IF
C
C2------CLEAR I/O FLAGS.
      IHDDFL=0
      IBUDFL=0
      ICBCFL=0
      DO 10 I=1,4
      DO 10 K=1,NLAY
      IOFLG(K,I)=0
10    CONTINUE
C
C3------IF OUTPUT CONTROL TIME STEP DOES NOT MATCH SIMULATION TIME STEP,
C3------WRITE MESSAGE THAT THERE IS NO OUTPUT CONTROL THIS TIME STEP,
C3------AND RETURN.
      IF(IPEROC.NE.KPER .OR. ITSOC.NE.KSTP) THEN
         WRITE(IOUT,11) KPER,KSTP
11       FORMAT(1X,/1X,'NO OUTPUT CONTROL FOR STRESS PERIOD',I3,
     1              '   TIME STEP',I3)
         RETURN
      END IF
C
C4------OUTPUT CONTROL TIME STEP MATCHES SIMULATION TIME STEP.
      WRITE(IOUT,12) IPEROC,ITSOC
12    FORMAT(1X,/1X,'OUTPUT CONTROL FOR STRESS PERIOD',I3,
     1              '   TIME STEP',I3)
C
C4A-----OUTPUT CONTROL MATCHES SIMULATION TIME.  READ NEXT OUTPUT
C4A-----RECORD; SKIP ANY BLANK LINES.
50    READ(INOC,'(A)',END=1000) LINE
      IF(LINE.EQ.' ') GO TO 50
C
C4A1----LOOK FOR "PERIOD", WHICH TERMINATES OUTPUT CONTROL FOR CURRENT
C4A1----TIME STEP.  IF FOUND, DECODE TIME STEP FOR NEXT OUTPUT.
      LLOC=1
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
      IF(LINE(ISTART:ISTOP).EQ.'PERIOD') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IPEROC,R,IOUT,INOC)
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).NE.'STEP') GO TO 2000
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,ITSOC,R,IOUT,INOC)
         RETURN
C
C4A2----LOOK FOR "PRINT", WHICH MAY REFER TO "BUDGET", "HEAD", OR
C4A2----"DRAWDOWN".
      ELSE IF(LINE(ISTART:ISTOP).EQ.'PRINT') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).EQ.'BUDGET') THEN
            WRITE(IOUT,53)
53          FORMAT(4X,'PRINT BUDGET')
            IBUDFL=1
         ELSE IF(LINE(ISTART:ISTOP).EQ.'HEAD') THEN
            CALL SBAS5L(1,LINE,LLOC,IOFLG,NLAY,IOUT,'PRINT HEAD',
     1              INOC)
            IHDDFL=1
         ELSE IF(LINE(ISTART:ISTOP).EQ.'DRAWDOWN') THEN
            CALL SBAS5L(2,LINE,LLOC,IOFLG,NLAY,IOUT,
     1              'PRINT DRAWDOWN',INOC)
            IHDDFL=1
         ELSE
            GO TO 2000
         END IF
C
C4A3----LOOK FOR "SAVE", WHICH MAY REFER TO "BUDGET", "HEAD", OR
C4A3----"DRAWDOWN".
      ELSE IF(LINE(ISTART:ISTOP).EQ.'SAVE') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
         IF(LINE(ISTART:ISTOP).EQ.'BUDGET') THEN
            WRITE(IOUT,57)
57          FORMAT(4X,'SAVE BUDGET')
            ICBCFL=IBDOPT
         ELSE IF(LINE(ISTART:ISTOP).EQ.'HEAD') THEN
            CALL SBAS5L(3,LINE,LLOC,IOFLG,NLAY,IOUT,'SAVE HEAD',INOC)
            IHDDFL=1
         ELSE IF(LINE(ISTART:ISTOP).EQ.'DRAWDOWN') THEN
            CALL SBAS5L(4,LINE,LLOC,IOFLG,NLAY,IOUT,'SAVE DRAWDOWN',
     1          INOC)
            IHDDFL=1
         ELSE
            GO TO 2000
         END IF
C
C4A4----WHEN NO KNOWN ALPHABETIC WORDS ARE FOUND, THERE IS AN ERROR.
      ELSE
         GO TO 2000
C
C4B-----AFTER SUCCESSFULLY DECODING ONE RECORD, READ ANOTHER.
      END IF
      GO TO 50
C
C5------END OF FILE WHILE READING AN OUTPUT CONTROL RECORD, SO THERE
C5------WILL BE NO FURTHER OUTPUT.  SET IPEROC AND ITSOC HIGH ENOUGH
C5------THAT THE MODEL TIME WILL NEVER MATCH THEM.
1000  IPEROC=9999
      ITSOC=9999
      RETURN
C
C6------ERROR DECODING ALPHABETIC INPUT STRUCTURE.
2000  WRITE(IOUT,2001) LINE
2001  FORMAT(1X,/1X,'ERROR READING OUTPUT CONTROL INPUT DATA:'/1X,A80)
      STOP
      END
      SUBROUTINE SBAS5L(IPOS,LINE,LLOC,IOFLG,NLAY,IOUT,LABEL,INOC)
C
C
C-----VERSION 1453 14FEB1994 SBAS5L
C     ******************************************************************
C     WHEN USING ALPHABETIC OUTPUT CONTROL, DECODE LAYER
C     NUMBERS FOR PRINTING OR SAVING HEAD OR DRAWDOWN
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      implicit double precision (a-h,o-z)
      DIMENSION IOFLG(NLAY,4)
      CHARACTER*80 LINE
      CHARACTER*(*) LABEL
      DIMENSION LAYER(200)
C     ------------------------------------------------------------------
C
C1------INITIALIZE COUNTER FOR NUMBER OF LAYERS FOR WHICH OUTPUT IS
C1------SPECIFIED.
      NSET=0
C
C2------CHECK FOR A VALID LAYER NUMBER.  WHEN FOUND, SET FLAG AND
C2------REPEAT.
10    CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,L,R,-1,INOC)
      IF(L.GT.0 .AND. L.LE.NLAY) THEN
         NSET=NSET+1
         LAYER(NSET)=L
         IOFLG(L,IPOS)=1
         GO TO 10
      END IF
C
C3------DONE CHECKING FOR LAYER NUMBERS.  IF NO LAYER NUMBERS WERE
C3------FOUND, SET FLAGS FOR ALL LAYERS.
      IF(NSET.EQ.0) THEN
         DO 110 K=1,NLAY
         IOFLG(K,IPOS)=1
110      CONTINUE
         WRITE(IOUT,111) LABEL
111      FORMAT(4X,A,' FOR ALL LAYERS')
C
C4------IF ONE OR MORE LAYER NUMBERS WERE FOUND, PRINT THE NUMBERS.
      ELSE
         WRITE(IOUT,112) LABEL,(LAYER(M),M=1,NSET)
112      FORMAT(4X,A,' FOR LAYERS:',(1X,15I3))
      END IF
C
C5------RETURN.
      RETURN
      END
      SUBROUTINE SBAS5O(INUNIT,INBAS,IOUT,IUNIT,CUNIT,ilstfil)
C
C-----VERSION 0943 18MAR1996 SBAS5O
C     ******************************************************************
C     OPEN FILES.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
	USE GWM1BAS1, ONLY : GWMOUT
	USE OFCMOD, ONLY : optinf,tempf
C
      implicit double precision (a-h,o-z)
      DIMENSION IUNIT(40)
      CHARACTER*4 CUNIT(40)
      CHARACTER*80 LINE
      CHARACTER*11 FMTARG
c
c--modofc-input/output 
c         ilstfil - if zero on input then this is first call to SBAS50
c                   on output contains MODFLOW listing file unit number
C     ---------------------------------------------------------------
C
C1------INITIALIZE CONSTANTS.  ILIST IS SET TO 1 ONCE THE LISTING
C1------FILE HAS BEEN OPENED; UNTIL THEN ERROR MESSAGES ARE WRITTEN
C1------ TO "*" UNIT.
      INBAS=0
c--modofc-iout is set in MODOFC
c     IOUT=0
      ILIST=0
      DO 5 I=1,40
      IUNIT(I)=0
5     CONTINUE
C
C2------READ A LINE; IGNORE BLANK LINES AND PRINT COMMENT LINES.
10    READ(INUNIT,'(A)',END=1000) LINE
      IF(LINE.EQ.' ') GO TO 10
      IF(LINE(1:1).EQ.'#') THEN
        IF(ILIST.NE.0) WRITE(IOUT,'(A)') LINE
        GO TO 10
      END IF
C
C3------DECODE THE FILE TYPE AND UNIT NUMBER.
      LLOC=1
      CALL URWORDd(LINE,LLOC,ITYP1,ITYP2,1,N,R,IOUT,INUNIT)
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IU,R,IOUT,INUNIT)
C
C4------CHECK FOR A VALID FILE TYPE.
      FMTARG='FORMATTED'
C
C4A-----FIRST ENTRY MUST BE FILE-TYPE "LIST".
      IF(ILIST.EQ.0) THEN
         IF(LINE(ITYP1:ITYP2).NE.'LIST') THEN
            WRITE(*,*) ' FIRST ENTRY IN NAME FILE MUST BE "LIST".'
            STOP
         END IF
c
c--modofc-save the MODFLOW listing file
	 il=iu
c
c--modofc-set the output unit number to the listing file
c         if iout is not zero, output will go to temporary file
	 if(iout.eq.0) iout=iu
c
c        IOUT=IU
C
C4B-----CHECK FOR "BAS" FILE TYPE.
      ELSE IF(LINE(ITYP1:ITYP2).EQ.'BAS') THEN
         INBAS=IU
C
C4C-----CHECK FOR "UNFORMATTED" FILE TYPE.
      ELSE IF(LINE(ITYP1:ITYP2).EQ.'DATA(BINARY)') THEN
         FMTARG='UNFORMATTED'
C
C4D-----CHECK FOR "FORMATTED FILE TYPE.
      ELSE IF(LINE(ITYP1:ITYP2).EQ.'DATA') THEN
         FMTARG='FORMATTED'
C
C4E-----CHECK FOR MAJOR OPTIONS.
      ELSE
         DO 20 I=1,40
            IF(LINE(ITYP1:ITYP2).EQ.CUNIT(I)) THEN
               IUNIT(I)=IU
               GO TO 30
            END IF
20       CONTINUE
         WRITE(IOUT,21) LINE(ITYP1:ITYP2)
21       FORMAT(1X,'ILLEGAL FILE TYPE IN NAME FILE: ',A)
         STOP
30       CONTINUE
      END IF
C
C5------DETERMINE FILE NAME AND THE ACCESS METHOD (DIRECT OR
C5------SEQUENTIAL).  WRITE THE FILE NAME IF THE FILE IS NOT THE
C5------LISTING FILE.  THEN OPEN THE FILE.
      CALL URWORDd(LINE,LLOC,INAM1,INAM2,0,N,R,IOUT,INUNIT)
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INUNIT)
      IF(LINE(ISTART:ISTOP).EQ.'DIRECT') THEN
         CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IRECL,R,IOUT,INUNIT)
         IF(ILIST.NE.0) WRITE(IOUT,35) LINE(INAM1:INAM2),
     1        LINE(ITYP1:ITYP2),IU,IRECL
35       FORMAT(1X,/1X,'OPENING ',A,/
     1     1X,'FILE TYPE:',A,'   UNIT',I4,'   DIRECT ACCESS',I10)
c
c--modofc-if this is first call to SBAS50 then open file
	   if(ilstfil.eq.0) 
     1         OPEN(UNIT=IU,FILE=LINE(INAM1:INAM2),FORM=FMTARG,
     2         ACCESS='DIRECT',RECL=IRECL)
      ELSE
         IF(ILIST.NE.0) WRITE(IOUT,36) LINE(INAM1:INAM2),
     1        LINE(ITYP1:ITYP2),IU
36       FORMAT(1X,/1X,'OPENING ',A,/
     1     1X,'FILE TYPE:',A,'   UNIT',I4)
c
c--modofc-if this is first call to SBAS50 then open file
         if(ilstfil.eq.0)
     1         OPEN(UNIT=IU,FILE=LINE(INAM1:INAM2),FORM=FMTARG,
     2         ACCESS='SEQUENTIAL')
      END IF
C
C6------IF THE OPENED FILE IS THE LISTING FILE, WRITE ITS NAME.
C6------GO BACK AND READ NEXT RECORD.
      IF(ILIST.EQ.0) WRITE(IOUT,37) LINE(INAM1:INAM2),IU
37    FORMAT(1X,'LISTING FILE: ',A,/25X,'UNIT',I4)
      ILIST=1
      GO TO 10
C
C7------END OF NAME FILE.  RETURN PROVIDED THAT LISTING FILE AND BAS
C7------FILES HAVE BEEN OPENED.
1000  IF(ILIST.EQ.0) THEN
         WRITE(*,*) ' NAME FILE IS EMPTY.'
         STOP
      ELSE IF(INBAS.EQ.0) THEN
         WRITE(IOUT,*) ' BAS PACKAGE FILE HAS NOT BEEN OPENED.'
         STOP
      END IF
c
c--modofc-the names file remains open for later calls to MODFLOW
c     CLOSE(UNIT=INUNIT)
c
c--modofc-retain the unit number of the MODFLOW list file
      ilstfil=il
c
      RETURN
C
      END
C=======================================================================
      INTEGER FUNCTION NONB_LEN(CHARVAR,LENGTH)
C     ******************************************************************
C     FUNCTION TO RETURN NON-BLANK LENGTH OF CONTENTS OF A CHARACTER
C     VARIABLE
C     ******************************************************************
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
C
C Variable list:
C   CHARVAR  = CHARACTER VARIABLE OF INTEREST
C   LENGTH   = DIMENSIONED LENGTH OF CHARVAR
C
      CHARACTER*(*) CHARVAR,C*1
      INTEGER LENGTH
C     ------------------------------------------------------------------
C
      C = ' '
      K = LENGTH+1
C
      DO 20 WHILE (C.EQ.' ' .AND. K.GE.1)
        K = K-1
        C = CHARVAR(K:K)
 20   CONTINUE
C
      NONB_LEN = K
C
      RETURN
      END
      SUBROUTINE SGWF1BAS6I(NLAY,IOFLG,INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,
     1   IDDNUN,IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,IFREFM,
     2   IBOUUN,LBBOSV,CBOUFM,IAUXSV,RESETDD,RESETDDNEXT)
C
C-----VERSION 05MAY2000 SGWF1BAS6I
C     ******************************************************************
C     SET UP OUTPUT CONTROL.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      DIMENSION IOFLG(NLAY,5)
      CHARACTER*20 CHEDFM,CDDNFM,CBOUFM
      CHARACTER*200 LINE
      LOGICAL RESETDD, RESETDDNEXT
C     ------------------------------------------------------------------
C
C1------ASSIGN DEFAULT VALUES.
      CHEDFM=' '
      CDDNFM=' '
      CBOUFM='(20I4)'
      IHEDFM=0
      IDDNFM=0
      IHEDUN=0
      IDDNUN=0
      IBOUUN=0
      IBDOPT=1
      LBHDSV=0
      LBDDSV=0
      LBBOSV=0
      IAUXSV=0
      RESETDD=.FALSE.
      RESETDDNEXT=.FALSE.
C
C2------TEST OUTPUT CONTROL INPUT UNIT TO SEE IF OUTPUT CONTROL IS
C2------ACTIVE.
      IF(INOC.LE.0) THEN
C
C2A-----OUTPUT CONTROL IS INACTIVE. PRINT A MESSAGE LISTING DEFAULTS.
         WRITE(IOUT, 41)
   41    FORMAT(1X,/1X,'DEFAULT OUTPUT CONTROL',/1X,
     1   'THE FOLLOWING OUTPUT COMES AT THE END OF EACH STRESS PERIOD:')
         WRITE(IOUT, 42)
   42    FORMAT(1X,'TOTAL VOLUMETRIC BUDGET')
         WRITE(IOUT, 43)
   43    FORMAT(1X,10X,'HEAD')
C
C2B-----SET DEFAULT FLAGS IN IOFLG SO THAT HEAD IS PRINTED FOR
C2B-----EVERY LAYER.
         DO 80 K=1,NLAY
         IOFLG(K,1)=1
         IOFLG(K,2)=0
         IOFLG(K,3)=0
         IOFLG(K,4)=0
         IOFLG(K,5)=0
   80    CONTINUE
         GO TO 1000
      END IF
C
C3------OUTPUT CONTROL IS ACTIVE.  READ FIRST RECORD AND DECODE FIRST
C3------WORD.  MUST USE URWORDd IN CASE FIRST WORD IS ALPHABETIC.
      CALL URDCOM(INOC,IOUT,LINE)
      LLOC=1
      CALL URWORDd(LINE,LLOC,ISTART,ISTOP,1,N,R,IOUT,INOC)
C
C4------TEST FOR NUMERIC OUTPUT CONTROL.  FIRST WORD WILL NOT BE
C4------"PERIOD", "HEAD", "DRAWDOWN", OR "COMPACT".
      IF(LINE(ISTART:ISTOP).NE.'PERIOD' .AND. LINE(ISTART:ISTOP).NE.
     1     'HEAD' .AND. LINE(ISTART:ISTOP).NE.'DRAWDOWN' .AND.
     2     LINE(ISTART:ISTOP).NE.'COMPACT' .AND.
     3     LINE(ISTART:ISTOP).NE.'IBOUND') THEN
C4A-----NUMERIC OUTPUT CONTROL.  DECODE THE INITIAL RECORD ACCORDINGLY.
         WRITE(IOUT,102)
  102    FORMAT(1X,/1X,'OUTPUT CONTROL IS SPECIFIED EVERY TIME STEP')
         IF(IFREFM.EQ.0) THEN
            READ(LINE,'(4I10)') IHEDFM,IDDNFM,IHEDUN,IDDNUN
         ELSE
            LLOC=1
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDFM,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNFM,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IHEDUN,R,IOUT,INOC)
            CALL URWORDd(LINE,LLOC,ISTART,ISTOP,2,IDDNUN,R,IOUT,INOC)
         END IF
         WRITE(IOUT,103) IHEDFM,IDDNFM
  103    FORMAT(1X,'HEAD PRINT FORMAT CODE IS',I4,
     1     '    DRAWDOWN PRINT FORMAT CODE IS',I4)
         WRITE(IOUT,104) IHEDUN,IDDNUN
  104    FORMAT(1X,'HEADS WILL BE SAVED ON UNIT ',I4,
     1     '    DRAWDOWNS WILL BE SAVED ON UNIT ',I4)
         IPEROC=-1
         ITSOC=-1
      ELSE
C4B-----ALPHABETIC OUTPUT CONTROL.  CALL MODULE TO READ INITIAL RECORDS.
C         CALL SGWF1BAS6J(INOC,IOUT,IHEDFM,IDDNFM,IHEDUN,IDDNUN,
C     1         IPEROC,ITSOC,CHEDFM,CDDNFM,IBDOPT,LBHDSV,LBDDSV,
C     2         LINE,LLOC,ISTART,ISTOP,IBOUUN,LBBOSV,CBOUFM,IAUXSV,
C     3         RESETDDNEXT)
      END IF
C
C5------RETURN.
 1000 RETURN
      END
      SUBROUTINE URWORD(LINE,ICOL,ISTART,ISTOP,NCODE,N,R,IOUT,IN)
C
C
C-----VERSION 1003 05AUG1992 URWORDd
c  this version uses a single precision R
C     ******************************************************************
C     ROUTINE TO EXTRACT A WORD FROM A LINE OF TEXT, AND OPTIONALLY
C     CONVERT THE WORD TO A NUMBER.
C        ISTART AND ISTOP WILL BE RETURNED WITH THE STARTING AND
C          ENDING CHARACTER POSITIONS OF THE WORD.
C        THE LAST CHARACTER IN THE LINE IS SET TO BLANK SO THAT IF ANY
C          PROBLEMS OCCUR WITH FINDING A WORD, ISTART AND ISTOP WILL
C          POINT TO THIS BLANK CHARACTER.  THUS, A WORD WILL ALWAYS BE
C          RETURNED UNLESS THERE IS A NUMERIC CONVERSION ERROR.  BE SURE
C          THAT THE LAST CHARACTER IN LINE IS NOT AN IMPORTANT CHARACTER
C          BECAUSE IT WILL ALWAYS BE SET TO BLANK.
C        A WORD STARTS WITH THE FIRST CHARACTER THAT IS NOT A SPACE OR
C          COMMA, AND ENDS WHEN A SUBSEQUENT CHARACTER THAT IS A SPACE
C          OR COMMA.  NOTE THAT THESE PARSING RULES DO NOT TREAT TWO
C          COMMAS SEPARATED BY ONE OR MORE SPACES AS A NULL WORD.
C        FOR A WORD THAT BEGINS WITH "'", THE WORD STARTS WITH THE
C          CHARACTER AFTER THE QUOTE AND ENDS WITH THE CHARACTER
C          PRECEDING A SUBSEQUENT QUOTE.  THUS, A QUOTED WORD CAN
C          INCLUDE SPACES AND COMMAS.  THE QUOTED WORD CANNOT CONTAIN
C          A QUOTE CHARACTER.
C        IF NCODE IS 1, THE WORD IS CONVERTED TO UPPER CASE.
C        IF NCODE IS 2, THE WORD IS CONVERTED TO AN INTEGER.
C        IF NCODE IS 3, THE WORD IS CONVERTED TO A REAL NUMBER.
C        NUMBER CONVERSION ERROR IS WRITTEN TO UNIT IOUT IF IOUT IS
C          POSITIVE; ERROR IS WRITTEN TO DEFAULT OUTPUT IF IOUT IS 0;
C          NO ERROR MESSAGE IS WRITTEN IF IOUT IS NEGATIVE.
C     ******************************************************************
C
C        SPECIFICATIONS:
C     ------------------------------------------------------------------
      CHARACTER*(*) LINE
      CHARACTER*20 RW,STRING
C     ------------------------------------------------------------------
C
C1------Set last char in LINE to blank and set ISTART and ISTOP to point
C1------to this blank as a default situation when no word is found.  If
C1------starting location in LINE is out of bounds, do not look for a
C1------word.
      LINLEN=LEN(LINE)
      LINE(LINLEN:LINLEN)=' '
      ISTART=LINLEN
      ISTOP=LINLEN
      LINLEN=LINLEN-1
      IF(ICOL.LT.1 .OR. ICOL.GT.LINLEN) GO TO 100
C
C2------Find start of word, which is indicated by first character that
C2------is not a blank and not a comma.
      DO 10 I=ICOL,LINLEN
      IF(LINE(I:I).NE.' ' .AND. LINE(I:I).NE.',') GO TO 20
10    CONTINUE
      ICOL=LINLEN+1
      GO TO 100
C
C3------Found start of word.  Look for end.
C3A-----When word is quoted, only a quote can terminate it.
20    IF(LINE(I:I).EQ.'''') THEN
         I=I+1
         IF(I.LE.LINLEN) THEN
            DO 25 J=I,LINLEN
            IF(LINE(J:J).EQ.'''') GO TO 40
25          CONTINUE
         END IF
C
C3B-----When word is not quoted, space or comma will terminate.
      ELSE
         DO 30 J=I,LINLEN
         IF(LINE(J:J).EQ.' ' .OR. LINE(J:J).EQ.',') GO TO 40
30       CONTINUE
      END IF
C
C3C-----End of line without finding end of word; set end of word to
C3C-----end of line.
      J=LINLEN+1
C
C4------Found end of word; set J to point to last character in WORD and
C-------set ICOL to point to location for scanning for another word.
40    ICOL=J+1
      J=J-1
      IF(J.LT.I) GO TO 100
      ISTART=I
      ISTOP=J
C
C5------Convert word to upper case and RETURN if NCODE is 1.
      IF(NCODE.EQ.1) THEN
         IDIFF=ICHAR('a')-ICHAR('A')
         DO 50 K=ISTART,ISTOP
            IF(LINE(K:K).GE.'a' .AND. LINE(K:K).LE.'z')
     1             LINE(K:K)=CHAR(ICHAR(LINE(K:K))-IDIFF)
50       CONTINUE
         RETURN
      END IF
C
C6------Convert word to a number if requested.
100   IF(NCODE.EQ.2 .OR. NCODE.EQ.3) THEN
         RW=' '
         L=20-ISTOP+ISTART
         IF(L.LT.1) GO TO 200
         RW(L:20)=LINE(ISTART:ISTOP)
         IF(NCODE.EQ.2) READ(RW,'(I20)',ERR=200) N
         IF(NCODE.EQ.3) READ(RW,'(F20.0)',ERR=200) R
      END IF
      RETURN
C
C7------Number conversion error.
200   IF(NCODE.EQ.3) THEN
         STRING= 'A REAL NUMBER'
         L=13
      ELSE
         STRING= 'AN INTEGER'
         L=10
      END IF
C
C7A-----If output unit is negative, set last character of string to 'E'.
      IF(IOUT.LT.0) THEN
         N=0
         R=0.
         LINE(LINLEN+1:LINLEN+1)='E'
         RETURN
C
C7B-----If output unit is positive; write a message to output unit.
      ELSE IF(IOUT.GT.0) THEN
         IF(IN.GT.0) THEN
            WRITE(IOUT,201) IN,LINE(ISTART:ISTOP),STRING(1:L),LINE
         ELSE
            WRITE(IOUT,202) LINE(ISTART:ISTOP),STRING(1:L),LINE
         END IF
201      FORMAT(1X,/1X,'FILE UNIT',I4,' : ERROR CONVERTING "',A,
     1       '" TO ',A,' IN LINE:',/1X,A)
202      FORMAT(1X,/1X,'KEYBOARD INPUT : ERROR CONVERTING "',A,
     1       '" TO ',A,' IN LINE:',/1X,A)
C
C7C-----If output unit is 0; write a message to default output.
      ELSE
         IF(IN.GT.0) THEN
            WRITE(*,201) IN,LINE(ISTART:ISTOP),STRING(1:L),LINE
         ELSE
            WRITE(*,202) LINE(ISTART:ISTOP),STRING(1:L),LINE
         END IF
      END IF
C
C7D-----STOP after writing message.
      STOP
      END



